library(shiny)
library(ggplot2)
library(dplyr)
library(plotly)

# Definir la interfaz de usuario
ui <- fluidPage(
  titlePanel("Modelo de Supervivencia de Makeham"),
  sidebarLayout(
    sidebarPanel(
      h4("Parámetros del Modelo"),
      numericInput("lo", "Tamaño de Cohorte Inicial (l₀):", 
                   value = 1000, min = 1, max = 100000),
      numericInput("edad_max", "Edad Máxima W:", 
                   value = 120, min = 50, max = 150),
      numericInput("S", "Parámetro S:", 
                   value = 0.98, min = 0.8, max = 0.999, step = 0.001),
      numericInput("g", "Parámetro g:", 
                   value = 0.99, min = 0.8, max = 0.999, step = 0.001),
      numericInput("c", "Parámetro c:", 
                   value = 1.03, min = 1.001, max = 1.2, step = 0.001),
      sliderInput("x_range", "Rango de edades a visualizar:",
                  min = 0, max = 150, value = c(0, 100)),
      actionButton("calcular", "Calcular", class = "btn-primary"),
      br(), br(),
      h4("Valores de Ejemplo Rápidos"),
      actionButton("ejemplo1", "Ejemplo 1: l₀=1000, S=0.98, g=0.99, c=1.03"),
      actionButton("ejemplo2", "Ejemplo 2: l₀=10000, S=0.99, g=0.97, c=1.02"),
      actionButton("ejemplo3", "Ejemplo 3: l₀=5000, S=0.995, g=0.995, c=1.01")
    ),
    
    mainPanel(
      tabsetPanel(
        tabPanel("Gráfica de Supervivencia",
                 plotlyOutput("grafica_supervivencia"),
                 br(),
                 h4("Ecuación del Modelo de Makeham"),
                 withMathJax("$$l_x = l_0 \\cdot S^x \\cdot g^{(c^x - 1)}$$")),
        
        tabPanel("Tabla de Vida",
                 downloadButton("descargar", "Descargar Tabla"),
                 br(), br(),
                 dataTableOutput("tabla_vida")),
        
        tabPanel("Análisis",
                 fluidRow(
                   column(6,
                          h4("Estadísticas Descriptivas"),
                          tableOutput("estadisticas")),
                   column(6,
                          h4("Probabilidades de Supervivencia"),
                          tableOutput("probabilidades"))
                 ),
                 fluidRow(
                   column(12,
                          h4("Tasas Instantáneas de Mortalidad"),
                          plotlyOutput("grafica_mortalidad"))
                 )),
        
        tabPanel("Comparación de Parámetros",
                 fluidRow(
                   column(6,
                          h4("Efecto del parámetro S"),
                          plotlyOutput("grafica_S")),
                   column(6,
                          h4("Efecto del parámetro g"),
                          plotlyOutput("grafica_g"))
                 ),
                 fluidRow(
                   column(6,
                          h4("Efecto del parámetro c"),
                          plotlyOutput("grafica_c"))
                 )),
        
        tabPanel("Información del Modelo",
                 h3("Modelo de Supervivencia de Makeham"),
                 p("Este modelo utiliza la función:"),
                 withMathJax("$$l_x = l_0 \\cdot S^x \\cdot g^{(c^x - 1)}$$"),
                 p("Donde:"),
                 tags$ul(
                   tags$li(withMathJax("$l_x$: Número de supervivientes a la edad $x$")),
                   tags$li(withMathJax("$l_0$: Tamaño inicial de la cohorte")),
                   tags$li(withMathJax("$S$: Parámetro de supervivencia base ($S < 1$)")),
                   tags$li(withMathJax("$g$: Parámetro relacionado con la mortalidad juvenil ($g < 1$)")),
                   tags$li(withMathJax("$c$: Parámetro de crecimiento de la mortalidad ($c > 1$)"))
                 ),
                 p("Interpretación de los parámetros:"),
                 tags$ul(
                   tags$li("S: Representa la mortalidad constante a todas las edades"),
                   tags$li("g: Controla la mortalidad en edades jóvenes"),
                   tags$li("c: Controla el crecimiento exponencial de la mortalidad con la edad")
                 ),
                 p("Características del modelo:"),
                 tags$ul(
                   tags$li("Captura tanto mortalidad constante como creciente con la edad"),
                   tags$li("Más flexible que el modelo de Gompertz"),
                   tags$li("juanin ")
                 ))
      )
    )
  )
)

# Definir el servidor
server <- function(input, output, session) {
  
  # Reactive values para almacenar resultados
  valores <- reactiveValues(datos = NULL)
  
  # Función del modelo de Makeham
  modelo_makeham <- function(x, lo, S, g, c) {
    lo * (S^x) * (g^(c^x - 1))
  }
  
  # Calcular cuando se presiona el botón
  observeEvent(input$calcular, {
    calcular_modelo()
  })
  
  # Ejemplos predefinidos
  observeEvent(input$ejemplo1, {
    updateNumericInput(session, "lo", value = 1000)
    updateNumericInput(session, "S", value = 0.98)
    updateNumericInput(session, "g", value = 0.99)
    updateNumericInput(session, "c", value = 1.03)
    calcular_modelo()
  })
  
  observeEvent(input$ejemplo2, {
    updateNumericInput(session, "lo", value = 10000)
    updateNumericInput(session, "S", value = 0.99)
    updateNumericInput(session, "g", value = 0.97)
    updateNumericInput(session, "c", value = 1.02)
    calcular_modelo()
  })
  
  observeEvent(input$ejemplo3, {
    updateNumericInput(session, "lo", value = 5000)
    updateNumericInput(session, "S", value = 0.995)
    updateNumericInput(session, "g", value = 0.995)
    updateNumericInput(session, "c", value = 1.01)
    calcular_modelo()
  })
  
  # Función para calcular el modelo
  calcular_modelo <- function() {
    req(input$lo, input$S, input$g, input$c, input$edad_max)
    
    # Validar parámetros
    if (input$S >= 1) {
      showNotification("El parámetro S debe ser menor que 1", type = "error")
      return()
    }
    if (input$g >= 1) {
      showNotification("El parámetro g debe ser menor que 1", type = "error")
      return()
    }
    if (input$c <= 1) {
      showNotification("El parámetro c debe ser mayor que 1", type = "error")
      return()
    }
    
    # Crear secuencia de edades
    edades <- 0:input$edad_max
    
    # Calcular lx para cada edad
    lx <- sapply(edades, modelo_makeham, 
                 lo = input$lo, S = input$S, g = input$g, c = input$c)
    
    # Crear data frame con resultados
    datos <- data.frame(
      Edad = edades,
      lx = lx,
      dx = c(-diff(lx), NA),  # Defunciones entre x y x+1
      qx = c(ifelse(lx[-length(lx)] > 0, 
                    -diff(lx)/lx[-length(lx)], NA), NA)  # Probabilidad de muerte
    )
    
    # Calcular probabilidad de supervivencia
    datos$px <- 1 - datos$qx
    
    # Calcular fuerza de mortalidad (aproximada)
    datos$mu_x <- -log(datos$px)
    
    valores$datos <- datos
  }
  
  # Gráfica de supervivencia principal
  output$grafica_supervivencia <- renderPlotly({
    req(valores$datos)
    
    datos_filtrados <- valores$datos %>%
      filter(Edad >= input$x_range[1], Edad <= input$x_range[2])
    
    p <- ggplot(datos_filtrados, aes(x = Edad, y = lx)) +
      geom_line(color = "steelblue", size = 1) +
      geom_point(color = "steelblue", size = 0.5) +
      labs(title = "Curva de Supervivencia - Modelo de Makeham",
           x = "Edad (x)",
           y = "Supervivientes (lₓ)") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p) %>%
      layout(hovermode = 'x unified')
  })
  
  # Gráfica de fuerza de mortalidad
  output$grafica_mortalidad <- renderPlotly({
    req(valores$datos)
    
    datos_filtrados <- valores$datos %>%
      filter(Edad >= input$x_range[1], Edad <= input$x_range[2],
             !is.na(mu_x))
    
    p <- ggplot(datos_filtrados, aes(x = Edad, y = mu_x)) +
      geom_line(color = "red", size = 1) +
      geom_point(color = "red", size = 0.5) +
      labs(title = "Fuerza de Mortalidad (μₓ)",
           x = "Edad (x)",
           y = "μₓ") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p) %>%
      layout(hovermode = 'x unified')
  })
  
  # Gráficas de sensibilidad de parámetros
  output$grafica_S <- renderPlotly({
    req(valores$datos)
    
    edades <- 0:100
    S_values <- c(0.97, 0.98, 0.99)
    
    datos_S <- data.frame()
    for(S_val in S_values) {
      lx_temp <- sapply(edades, modelo_makeham, 
                        lo = 1000, S = S_val, g = input$g, c = input$c)
      datos_S <- rbind(datos_S, 
                       data.frame(Edad = edades, lx = lx_temp, S = as.factor(S_val)))
    }
    
    p <- ggplot(datos_S, aes(x = Edad, y = lx, color = S)) +
      geom_line(size = 1) +
      labs(title = "Sensibilidad al parámetro S",
           x = "Edad (x)",
           y = "Supervivientes (lₓ)") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p)
  })
  
  output$grafica_g <- renderPlotly({
    req(valores$datos)
    
    edades <- 0:100
    g_values <- c(0.97, 0.98, 0.99)
    
    datos_g <- data.frame()
    for(g_val in g_values) {
      lx_temp <- sapply(edades, modelo_makeham, 
                        lo = 1000, S = input$S, g = g_val, c = input$c)
      datos_g <- rbind(datos_g, 
                       data.frame(Edad = edades, lx = lx_temp, g = as.factor(g_val)))
    }
    
    p <- ggplot(datos_g, aes(x = Edad, y = lx, color = g)) +
      geom_line(size = 1) +
      labs(title = "Sensibilidad al parámetro g",
           x = "Edad (x)",
           y = "Supervivientes (lₓ)") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p)
  })
  
  output$grafica_c <- renderPlotly({
    req(valores$datos)
    
    edades <- 0:100
    c_values <- c(1.02, 1.03, 1.04)
    
    datos_c <- data.frame()
    for(c_val in c_values) {
      lx_temp <- sapply(edades, modelo_makeham, 
                        lo = 1000, S = input$S, g = input$g, c = c_val)
      datos_c <- rbind(datos_c, 
                       data.frame(Edad = edades, lx = lx_temp, c = as.factor(c_val)))
    }
    
    p <- ggplot(datos_c, aes(x = Edad, y = lx, color = c)) +
      geom_line(size = 1) +
      labs(title = "Sensibilidad al parámetro c",
           x = "Edad (x)",
           y = "Supervivientes (lₓ)") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p)
  })
  
  # Tabla de vida
  output$tabla_vida <- renderDataTable({
    req(valores$datos)
    
    valores$datos %>%
      mutate(across(c(lx, dx), round, 2),
             across(c(qx, px, mu_x), round, 4))
  }, options = list(pageLength = 10))
  
  # Estadísticas descriptivas
  output$estadisticas <- renderTable({
    req(valores$datos)
    
    datos <- valores$datos
    
    # Esperanza de vida aproximada
    esperanza_vida <- sum(datos$lx[-1], na.rm = TRUE) / datos$lx[1]
    
    # Edad modal al fallecimiento
    edad_modal <- datos$Edad[which.max(datos$dx)]
    
    # Edad cuando lx se reduce a la mitad
    edad_mediana <- datos$Edad[which(datos$lx <= datos$lx[1]/2)[1]]
    
    data.frame(
      Estadística = c("Cohorte Inicial", 
                      "Esperanza de Vida",
                      "Edad Modal al Fallecimiento",
                      "Edad Mediana de Supervivencia",
                      "Supervivientes a 65 años",
                      "Supervivientes a 85 años"),
      Valor = c(input$lo,
                round(esperanza_vida, 2),
                edad_modal,
                ifelse(!is.na(edad_mediana), edad_mediana, "> Máxima"),
                round(datos$lx[datos$Edad == 65][1], 2),
                round(datos$lx[datos$Edad == 85][1], 2))
    )
  }, bordered = TRUE)
  
  # Probabilidades de supervivencia
  output$probabilidades <- renderTable({
    req(valores$datos)
    
    datos <- valores$datos
    
    # Encontrar valores para edades específicas
    lx_20 <- datos$lx[datos$Edad == 20][1]
    lx_65 <- datos$lx[datos$Edad == 65][1]
    lx_40 <- datos$lx[datos$Edad == 40][1]
    lx_80 <- datos$lx[datos$Edad == 80][1]
    
    data.frame(
      Probabilidad = c("20p0 (Llegar a 20 años)",
                       "45p20 (Llegar a 65 desde 20)",
                       "65p0 (Llegar a 65 años)",
                       "40p20 (Llegar a 60 desde 20)"),
      Valor = c(ifelse(!is.na(lx_20), 
                       round(lx_20/input$lo, 4), NA),
                ifelse(!is.na(lx_20) & !is.na(lx_65) & lx_20 > 0,
                       round(lx_65/lx_20, 4), NA),
                ifelse(!is.na(lx_65), 
                       round(lx_65/input$lo, 4), NA),
                ifelse(!is.na(lx_20) & !is.na(lx_40) & lx_20 > 0,
                       round(lx_40/lx_20, 4), NA))
    )
  }, bordered = TRUE)
  
  # Descargar tabla
  output$descargar <- downloadHandler(
    filename = function() {
      paste0("tabla_vida_makeham_", Sys.Date(), ".csv")
    },
    content = function(file) {
      write.csv(valores$datos, file, row.names = FALSE)
    }
  )
}

# Ejecutar la aplicación
shinyApp(ui = ui, server = server)