library(shiny)
library(ggplot2)
library(dplyr)

# UI
ui <- fluidPage(
  titlePanel("Modelo de Mortalidad Dormoy 2"),
  
  sidebarLayout(
    sidebarPanel(
      h4("Parámetros del Modelo"),
      sliderInput("s1", "S1 (supervivencia base):", 
                  min = 0.98, max = 0.999, value = 0.997, step = 0.001),
      sliderInput("s2", "S2 (aceleración edad):", 
                  min = 0.999, max = 0.99999, value = 0.9999, step = 0.00001),
      sliderInput("edad", "Edad actual (x):", 
                  min = 0, max = 100, value = 0),
      sliderInput("omega", "Edad máxima (ω):", 
                  min = 100, max = 200, value = 130),
      numericInput("radix", "Población inicial (l₀):", 
                   value = 100000, min = 1000, max = 1000000),
      actionButton("calcular", "Calcular")
    ),
    
    mainPanel(
      tabsetPanel(
        tabPanel("Gráficos",
                 plotOutput("grafico_supervivencia"),
                 plotOutput("grafico_esperanza"),
                 plotOutput("grafico_mortalidad")
        ),
        tabPanel("Resultados",
                 h4("Indicadores Clave"),
                 tableOutput("indicadores"),
                 h4("Tabla de Mortalidad (resumen)"),
                 tableOutput("tabla_resumen")
        ),
        tabPanel("Fórmulas",
                 withMathJax(),
                 h4("Modelo Dormoy 2:"),
                 helpText("$$l_x = l_0 \\cdot S_1^x \\cdot S_2^{x^2}$$"),
                 h4("Esperanza de vida:"),
                 helpText("$$e_x = \\frac{T_x}{l_x} = \\frac{\\sum_{t=x}^{\\omega} L_t}{l_x}$$"),
                 h4("Tasa instantánea de mortalidad:"),
                 helpText("$$\\mu_x = -\\frac{d}{dx} \\ln l_x = -\\ln S_1 - 2x \\ln S_2$$")
        )
      )
    )
  )
)

# Server
server <- function(input, output, session) {
  
  # Función para calcular tabla de mortalidad
  calcular_tabla <- reactive({
    req(input$calcular)
    
    S1 <- input$s1
    S2 <- input$s2
    omega <- input$omega
    radix <- input$radix
    
    edad <- 0:omega
    n <- length(edad)
    
    # Sobrevivientes
    l_x <- radix * (S1 ^ edad) * (S2 ^ (edad ^ 2))
    
    # Defunciones
    d_x <- -diff(l_x)
    d_x <- c(d_x, l_x[n])
    
    # Años-persona
    L_x <- numeric(n)
    for(i in 1:(n-1)) {
      L_x[i] <- (l_x[i] + l_x[i+1]) / 2
    }
    L_x[n] <- l_x[n] * 0.5
    
    # Años por vivir
    T_x <- rev(cumsum(rev(L_x)))
    
    # Esperanza de vida
    e_x <- T_x / l_x
    
    # Tasa instantánea de mortalidad
    mu_x <- -log(S1) - 2 * edad * log(S2)
    
    data.frame(
      edad = edad,
      l_x = l_x,
      d_x = d_x,
      L_x = L_x,
      T_x = T_x,
      e_x = e_x,
      mu_x = mu_x
    )
  })
  
  # Gráfico de función de supervivencia
  output$grafico_supervivencia <- renderPlot({
    tabla <- calcular_tabla()
    
    ggplot(tabla, aes(x = edad, y = l_x/input$radix)) +
      geom_line(color = "blue", linewidth = 1) +
      geom_vline(xintercept = input$edad, color = "red", linetype = "dashed") +
      labs(title = "Función de Supervivencia lₓ/l₀",
           x = "Edad (x)",
           y = "Proporción de Sobrevivientes") +
      theme_minimal() +
      scale_y_continuous(labels = scales::percent)
  })
  
  # Gráfico de esperanza de vida
  output$grafico_esperanza <- renderPlot({
    tabla <- calcular_tabla()
    
    ggplot(tabla, aes(x = edad, y = e_x)) +
      geom_line(color = "darkgreen", linewidth = 1) +
      geom_vline(xintercept = input$edad, color = "red", linetype = "dashed") +
      geom_point(data = tabla[tabla$edad == input$edad, ], 
                 color = "red", size = 3) +
      labs(title = paste("Esperanza de Vida eₓ (e₀ =", round(tabla$e_x[1], 2), "años)"),
           x = "Edad (x)",
           y = "Esperanza de Vida Restante") +
      theme_minimal()
  })
  
  # Gráfico de tasa instantánea de mortalidad
  output$grafico_mortalidad <- renderPlot({
    tabla <- calcular_tabla()
    
    ggplot(tabla, aes(x = edad, y = mu_x)) +
      geom_line(color = "red", linewidth = 1) +
      geom_vline(xintercept = input$edad, color = "blue", linetype = "dashed") +
      labs(title = "Tasa Instantánea de Mortalidad μₓ",
           x = "Edad (x)",
           y = "Fuerza de Mortalidad") +
      theme_minimal()
  })
  
  # Tabla de indicadores clave
  output$indicadores <- renderTable({
    tabla <- calcular_tabla()
    edad_actual <- input$edad + 1  # +1 porque índice empieza en 1
    
    data.frame(
      Indicador = c("Esperanza de vida al nacer (e₀)",
                    paste("Esperanza de vida a los", input$edad, "años (eₓ)"),
                    "Probabilidad de supervivencia a edad actual",
                    "Tasa instantánea de mortalidad a edad actual",
                    "Edad con máxima esperanza de vida restante"),
      Valor = c(
        round(tabla$e_x[1], 4),
        round(tabla$e_x[edad_actual], 4),
        paste0(round(tabla$l_x[edad_actual]/input$radix * 100, 2), "%"),
        round(tabla$mu_x[edad_actual], 6),
        tabla$edad[which.max(tabla$e_x)]
      )
    )
  }, bordered = TRUE)
  
  # Tabla resumen de mortalidad
  output$tabla_resumen <- renderTable({
    tabla <- calcular_tabla()
    
    edades_mostrar <- unique(c(0, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100))
    edades_mostrar <- edades_mostrar[edades_mostrar <= input$omega]
    
    tabla_resumen <- tabla[tabla$edad %in% edades_mostrar, 
                           c("edad", "l_x", "e_x", "mu_x")]
    
    tabla_resumen$l_x <- round(tabla_resumen$l_x)
    tabla_resumen$e_x <- round(tabla_resumen$e_x, 2)
    tabla_resumen$mu_x <- round(tabla_resumen$mu_x, 6)
    
    names(tabla_resumen) <- c("Edad", "Sobrevivientes", "Esperanza Vida", "μₓ")
    
    tabla_resumen
  }, bordered = TRUE)
}

# Run the application
shinyApp(ui = ui, server = server)