// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://grev.uv.es
// http://www.uv.es/agentes
//
//----------------------------------------------------------------------------

package jmadem;

import jason.asSyntax.Term;
import jason.asSyntax.NumberTermImpl;
import jason.asSyntax.Atom;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Vector;
import java.util.List;
import java.util.Iterator;


/** 
 * Data structure holding all the information related to a specific decision
 * problem.
 *
 * @author Francisco Grimaldo
 */
public class DecisionData 
{
	/** Decision id generator*/
	static private int id_generator = 1;
	
	/** Decision id */
	private int id;
	
	/**
	 * Agents involved in the decision-making.
	 */
	private Vector<Atom> agents;

	/**
	 * Allocations being considered as solutions for the decision problem.
	 */
	private Vector<Allocation> allocations;

	/**
	 * Names of the utility functions being considered.
	 */
	private Vector<Atom> utilityFunctions;
	 
	/**
	 * Utility weights representing the importance given to each point of view
	 * in the multi-modal agent decision making.
	 */
	private ConcurrentMap<Atom, Float> utilityWeights;
	
	/**
	 * Personal weights representing the internal attitude of the agent towards
	 * other individuals.
	 */
	private ConcurrentMap<Atom, Float> personalWeights;
	
	/** Social welfare considered by the agent */
	private Cuf welfare;
	
	/**
	 * Utility values received from the agents involved in the decision.
	 */
	private ConcurrentMap<Atom, Vector<Float>> utilityValues;

	
	/**
	  * DecisionData constructor.
	  *
	  * @param	ags		names of the agents involved in the decision
	  * @param	allocs	allocations being considered as solutions for the decision problem
	  * @param	uFuncs	names of the different utility functions to pay attention to 
	  * @param	pW		personal weights to be applied over external preferences
	  * @param	uW		utility weights in the multi-modal decision
	  * @param	sw		type of social welfare being considered 
	  */
	DecisionData(Vector<Atom> ags, Vector<Allocation> allocs, Vector<Atom> uFuncs,
				 ConcurrentMap<Atom, Float> pW, ConcurrentMap<Atom, Float> uW, Cuf sw)
	{
		// Assign decision id
		id = id_generator;
		if (id_generator == Integer.MAX_VALUE)
			id_generator = 1;
		else
			id_generator++;
		
		// Store current MADeM parameters 
		agents = ags;
		allocations = allocs;
		utilityFunctions = uFuncs;
		personalWeights = new ConcurrentHashMap<Atom, Float>();
		Iterator<Map.Entry<Atom, Float>> it = pW.entrySet().iterator();
		Map.Entry<Atom, Float> element;
		while (it.hasNext())
		{
			element = it.next();
			personalWeights.put((Atom) (element.getKey().clone()), new Float(element.getValue()));
		}		
		utilityWeights = new ConcurrentHashMap<Atom, Float>();
		it = uW.entrySet().iterator();
		while (it.hasNext())
		{
			element = it.next();
			utilityWeights.put((Atom) (element.getKey().clone()), new Float(element.getValue()));
		}		
		welfare = sw;
		utilityValues = new ConcurrentHashMap<Atom, Vector<Float>>();
	}
	
	/**
	  * Returns the identifier of the decision making.
	  *
	  * @return	decision id for the decision making
	  */
	public int getId()
	{
		return id;
	}
	
	/**
	  * Adds a bid to the MADeM decision data.
	  *
	  * @param	bidder	name of the bidder agent
	  * @param	uValues	utility values obtained by the set utility functions
	  *                 being considered for the different allocations
	  */
	public void addBid(String bidder, List<Term> uValues)
	{
		Vector<Float> uV = new Vector<Float>();
		Iterator<Term> it = uValues.iterator();
		while (it.hasNext())
			uV.add(new Float( ((NumberTermImpl) it.next()).solve() ));
		
		// Store utility values
		utilityValues.put(new Atom(bidder), uV);
	}
	
	/**
	  * Checks whether all agents have already bid and their bids have been received.  
	  *
	  * @return	<tt>true</tt> if all bids have been received, <tt>false</tt> otherwise
	  */
	public boolean areAllBidsReceived()
	{
		return (agents.size() == utilityValues.size());
	}

	/**
	  * Returns the name of the agents involved in the decision.  
	  *
	  * @return	the name of the agents into a <tt>Vector</tt>
	  */
	public Vector<Atom> getAgents()
	{
		return agents;
	}

	/**
	  * Returns the personal weights for the agents involved in the decision.  
	  *
	  * @return	the personal weights into a <tt>ConcurrentHashMap</tt>, where the 
	  *         keys of the map are the names of the agents
	  */
	public ConcurrentMap<Atom, Float> getPersonalWeights()
	{
		return personalWeights;
	}

	/**
	  * Returns the allocations considered as solutions for the decision problem.  
	  *
	  * @return	the allocations into a <tt>Vector</tt>
	  */
	public Vector<Allocation> getAllocations()
	{
		return allocations;
	}

		/**
	  * Returns the name of the utility functions used to express preference.  
	  *
	  * @return	the name of the utility functions into a <tt>Vector</tt>
	  */
	public Vector<Atom> getUtilityFunctions()
	{
		return utilityFunctions;
	}

	/**
	  * Returns the utility weights for the utility functions being used.  
	  *
	  * @return	the utility weights into a <tt>ConcurrentHashMap</tt>, where the 
	  *         keys of the map are the names of the utility functions
	  */
	public ConcurrentMap<Atom, Float> getUtilityWeights()
	{
		return utilityWeights;
	}

	/**
	  * Returns the social welfare applied to solve the decision problem.  
	  *
	  * @return	the social welfare for the decision
	  */
	public Cuf getWelfare()
	{
		return welfare;
	}

	/**
	  * Returns the utility values received from the agents involved in the decision.  
	  *
	  * @return	the utility values into a <tt>ConcurrentHashMap</tt>, where the 
	  *         keys of the map are the names of the bidders
	  */
	public ConcurrentMap<Atom, Vector<Float>> getUtilityValues()
	{
		return utilityValues;
	}
}

