// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://grev.uv.es
// http://www.uv.es/agentes
//
//----------------------------------------------------------------------------

package jmadem;

import jason.JasonException;
import jason.architecture.AgArch;
import jason.asSemantics.Message;
import jason.asSyntax.ASSyntax;
import jason.asSyntax.Atom;
import jason.asSyntax.Literal;
import jason.asSyntax.Term;
import jason.asSyntax.ListTermImpl;
import jason.asSyntax.NumberTermImpl;
import jason.mas2j.ClassParameters;
import jason.runtime.Settings;

import java.util.Set;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Vector;
import java.util.List;
import java.util.Iterator;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Logger;

/**
 * Types of Social Welfares available to MADeM agents.
 */
enum Cuf {UTILITARIAN, EGALITARIAN, ELITIST, NASH};


/**
  * Basic architecture for a Jason agent performing MADeM decisions.
  *
  * @author Francisco Grimaldo
  */
public class MADeMAgArch extends AgArch 
{
	/** Class logger */
	private Logger logger = Logger.getLogger(MADeMAgArch.class.getName());
	
	/** Agent name */
	private String agName;

	/** Utility functions repository */
	private ConcurrentMap<Atom, UtilityFunctionInt> utilityFunctions = new ConcurrentHashMap<Atom, UtilityFunctionInt>();
	
	/**
	 * Utility weights representing the importance given to each point of view
	 * in the multi-modal agent decision making.
	 */
	private ConcurrentMap<Atom, Float> utilityWeights = new ConcurrentHashMap<Atom, Float>();
	
	/**
	 * Personal weights representing the internal attitude of the agent towards
	 * other individuals.
	 */
	private ConcurrentMap<Atom, Float> personalWeights = new ConcurrentHashMap<Atom, Float>();
	
	/** Social welfare considered by the agent */
	private Cuf welfare;
	
	/** 
	 * Maximum waiting time for bids (in milliseconds). After timeout, the agent 
	 * decides according to the bids it has collected and its own preferences.
	 */
	private int tout;

	/**
	 * Data managed by MADeM for every decision being performed: id_decision, agents involved, 
	 * allocations, utility functions considered, bids...
	 */
	private ConcurrentMap<Integer, DecisionData> mademData = new ConcurrentHashMap<Integer, DecisionData>();
	
	
	/**
	  * MADeM agent initiator. Creates the agent class defined by agClass, 
	  * default is jason.asSemantics.Agent. The agent class will parse the source
	  * code, create the transition system (TS), ... 
	  *
	  * @param	agClass	agent class definition
	  * @param	bbPars	agent class parameters
	  * @param	asSrc	AgentSpeak source code
	  * @param	stts	MAS Runtime Settings for an Agent
	  * @throws	JasonException
	  */
    @Override
	public void initAg(String agClass, ClassParameters bbPars, String asSrc, Settings stts) throws JasonException 
	{
        super.initAg(agClass, bbPars, asSrc, stts);

        // Agent Name
		this.agName = getAgName();
        
		// Init logger 
		logger = Logger.getLogger(MADeMAgArch.class.getName() + "." + agName);

		// Default social welfare
		this.welfare = Cuf.ELITIST; 
		
		// Default timeout
		this.tout = 10000;		
    }

	/**
	  * Registers a new utility function in the MADeM agent.
	  *
	  * @param	uFuncName	name of the utility function to be added
	  * @return	<tt>true</tt> if the utility function was correctly added, <tt>false</tt> otherwise
	  */
	public boolean addUtilityFunction(String uFuncName)
	{
		UtilityFunctionInt u = null;
		
		try
		{
			u = (UtilityFunctionInt) (Class.forName(uFuncName).newInstance());
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return false;
		}
		
		if (utilityFunctions.put(new Atom(u.getId()), u) != null)
			logger.info("Utility function \"" + u.getId() + "\" has been updated.");
		else
			logger.fine("Utility function \"" + u.getId() + "\" has been registered.");

		return true;
	}

	/**
	  * Removes a utility function previously registered in the MADeM agent.  
	  *
	  * @param	uFuncName	name of the utility function to be removed
	  * @return	<tt>true</tt> if the utility function was correctly removed, <tt>false</tt> otherwise
	  */
	public boolean removeUtilityFunction(String uFuncName)
	{
		return (utilityFunctions.remove(new Atom(uFuncName)) == null);
	}

	/**
	  * Gets the names of the utility functions previously registered in the MADeM agent.  
	  *
	  * @return the utility function names in the form of a <tt>Set</tt> 
	  */
	public Set<Atom> getUtilityFunctionNames()
	{
		return utilityFunctions.keySet();
	}

	/**
	  * Starts a MADeM decision using the internal parameters of the agent.  
	  *
	  * @param	ags		names of the agents involved in the decision
	  * @param	allocs	allocations being considered as solutions for the decision problem
	  * @param	uFuncs	names of the different utility functions to pay attention to 
	  *
	  * @return	the identifier of the MADeM decision or <tt>-1</tt> when any error occurs 
	  */
	public int startMADeMDecision(ListTermImpl ags, ListTermImpl allocs, ListTermImpl uFuncs)
	{
		return startMADeMDecision(ags, allocs, uFuncs, this.personalWeights, this.utilityWeights, this.welfare, this.tout);
	}

	/**
	  * Starts a MADeM decision by specifying all its parameters.  
	  *
	  * @param	ags		names of the agents involved in the decision
	  * @param	allocs	allocations being considered as solutions for the decision problem
	  * @param	uFuncs	names of the different utility functions to pay attention to 
	  * @param	pW		personal weights to be applied over external preferences
	  * @param	uW		utility weights in the multi-modal decision
	  * @param	sw		type of social welfare being considered 
	  * @param	tout	timeout for getting bids (in milliseconds)
	  *
	  * @return	the identifier of the MADeM decision or <tt>-1</tt> when any error occurs 
	  */
	public int startMADeMDecision(ListTermImpl ags, ListTermImpl allocs, ListTermImpl uFuncs,
								  ConcurrentMap<Atom, Float> pW, ConcurrentMap<Atom, Float> uW, Cuf sw, int tout)
	{
		// Read the first three parameters into Vectors (Agents, Allocations, Utility Functions)
		Vector<Atom> v_ags = new Vector<Atom>();
		Vector<Allocation> v_allocs = new Vector<Allocation>();
		Vector<Atom> v_uFuncs = new Vector<Atom>();
		Iterator<Term> it = ags.getAsList().iterator();
		while (it.hasNext())
			v_ags.add((Atom) it.next());
		it = allocs.iterator();
		while (it.hasNext())
			v_allocs.add( new Allocation((ListTermImpl) it.next()) );
		it = uFuncs.iterator();
		while (it.hasNext())
			v_uFuncs.add((Atom) it.next());

		// Create and fill decision data
		DecisionData decision = new DecisionData(v_ags, v_allocs, v_uFuncs, pW, uW, sw);
		mademData.put(new Integer(decision.getId()), decision);

		// Prepare auction message to ask for preferences
		Literal auction = ASSyntax.createLiteral("madem_auction");
		auction.addTerm(new NumberTermImpl((double) decision.getId()));
		auction.addTerm(allocs);
		auction.addTerm(uFuncs);
		logger.fine("Auction message: " + auction);

		// Send auctions to bidders
		Message msg = new Message("tell",
								agName,
								agName,
								auction);
		for (int i = 0; i < v_ags.size(); i++)
		{
			if (v_ags.get(i).toString().equals(agName))
			{
				logger.fine("Obtaining personal preferences for " + auction);
				bid(agName, auction);
			}
			else
			{			
				msg.setReceiver(v_ags.get(i).toString());
				try
				{
					sendMsg(msg);
				}
				catch (Exception e)
				{
					logger.severe("Error sending message \"" + msg + "\" while starting MADeM.");
					e.printStackTrace();
	
					// Remove decision data
					mademData.remove(new Integer(decision.getId()));
					
					return -1;
				}
				
				logger.fine("Auctioned message sent to " + v_ags.get(i));
			}
		}
		
		// Wait for bids
		if (v_ags.size() != 0)
		{
			// Launch bid timeout handler
			Timer timer = new Timer();
			timer.schedule(new BidTimeout(decision.getId()), tout);
		}
		else  // No other agent involved in the decision
			winnerDeterminationProblem(decision.getId());

		return decision.getId();
	}
	
	/**
	  * Handles bid timeout for a specific decision making.  
	  */
	class BidTimeout extends TimerTask
	{
		int id_decision;	
		
		public BidTimeout(int id) {
			id_decision = id;
		}
		
		public void run() {
			if (mademData.containsKey(id_decision))
			{
				logger.fine("Bid timeout reached for decision \"" + id_decision);
				winnerDeterminationProblem(id_decision);
			}
		}
	}	

	/**
	  * Filters MADeM messages out of the agent Mailbox.  
	  */
	@Override
	public void checkMail()
	{
        super.checkMail(); // get the messages
        
        // check the MailBox (at TS) for org messages
        Iterator<Message> i = getTS().getC().getMailBox().iterator();
        while (i.hasNext()) 
		{
            Message m = i.next();
			Object body = m.getPropCont();
			
			//"madem_auction"
			if ( (body instanceof Literal) && ((Literal) body).getFunctor().equals("madem_auction") )
			{
				i.remove(); // the agent do not receive this message
				bid(m.getSender(), (Literal) body);
			}
			
			//"madem_bid"
			if ( (body instanceof Literal) && ((Literal) body).getFunctor().equals("madem_bid") )
			{
				i.remove(); // the agent do not receive this message
				processBid(m.getSender(), (Literal) body);
			}
		}
	}
	
	/**
	  * Changes bid timeout to a new positive value.  
	  *
	  * @param	new_tout	new timeout value (in milliseconds)
	  * @return	<tt>true</tt> if Timeout was correctly changed, <tt>false</tt> otherwise
	  */
	public boolean setTimeout(int new_tout)
	{
		if (new_tout > 0)
		{
			tout = new_tout;
			return true;
		}
		
		return false;
	}
	
	/**
	  * Gets bid timeout (in milliseconds).  
	  *
	  * @return	the bid timeout (in milliseconds) used by the MADeM agent
	  */
	public int getTimeout()
	{
		return tout;
	}

	/**
	  * Changes social welfare of the MADeM agent.  
	  *
	  * @param	new_sw	new social welfare
	  */
	public void setWelfare(Cuf new_sw)
	{
		welfare = new_sw;
	}

	/**
	  * Gets the social welfare of the MADeM agent.  
	  *
	  * @return	the name of the social welfare
	  */
	public String getWelfare()
	{
		return welfare.name();
	}

	/**
	  * Changes the personal weights of the MADeM agent towards the other agents.  
	  *
	  * @param	new_pw	new personal weights
	  * @return	<tt>true</tt> if Personal Weights were correctly changed, <tt>false</tt> otherwise
	  */
	public boolean setPersonalWeights(ConcurrentMap<Atom, Float> new_pw)
	{
		//// Complete transactions while changing personal weights.
		//// Comment synchronized code in order to fast changes when concurrency is low
		synchronized(personalWeights)
		{
			personalWeights.putAll(new_pw);
		}

		return true;
	}

	/**
	  * Gets the personal weights of the MADeM agent.  
	  *
	  * @return	the personal weights being used by de MADeM agent in the form of a <tt>ConcurrentMap</tt> 
	  */
	public ConcurrentMap<Atom, Float> getPersonalWeights()
	{
		return personalWeights;
	}

	/**
	  * Changes the utility weights of the MADeM agent.  
	  *
	  * @param	new_uw	new utility weights
	  * @return	<tt>true</tt> if Utility Weights were correctly changed, <tt>false</tt> otherwise
	  */
	public boolean setUtilityWeights(ConcurrentMap<Atom, Float> new_uw)
	{
		//// Complete transactions while changing utility weights.
		//// Comment synchronized code in order to fast changes when concurrency is low
		synchronized (utilityWeights)
		{
			utilityWeights.putAll(new_uw);
		}

		return true;
	}

	/**
	  * Gets the utility weights of the MADeM agent.  
	  *
	  * @return	the utility weights being used by de MADeM agent in the form of a <tt>ConcurrentMap</tt> 
	  */
	public ConcurrentMap<Atom, Float> getUtilityWeights()
	{
		return utilityWeights;
	}

	/**
	  * Bids over a set of allocations coming from an auctioneer according
	  * to the specified utility function.  
	  *
	  * @param	auctioneer	agent asking for preference
	  * @param	madem_auction	allocations and utility functions being considered
	  */
	public void bid(String auctioneer, Literal madem_auction)
	{
		float utilityValue = 0;
		Literal bidResult = ASSyntax.createLiteral("madem_bid");
		bidResult.addTerm(madem_auction.getTerm(0));
		List<Term> allocs = ((ListTermImpl) madem_auction.getTerm(1)).getAsList();
		ListTermImpl uFuncs = (ListTermImpl) madem_auction.getTerm(2);

		// Compute utility values
		ListTermImpl uValues = new ListTermImpl();
		for (int i = 0; i < allocs.size(); i++)
		{
			Allocation alloc = new Allocation((ListTermImpl) allocs.get(i));
			Iterator<Term> it_uFuncs = uFuncs.iterator();
			while (it_uFuncs.hasNext())
			{
				utilityValue = getUtility(auctioneer, alloc, (Atom) it_uFuncs.next());
				uValues.add(new NumberTermImpl((double) utilityValue));
			}
		}
		bidResult.addTerm(uValues);

		// Send bids to auctioneer
		if (agName.equals(auctioneer))
		{
			processBid(agName, bidResult);
		}
		else
		{
			Message msg = new Message("tell", agName, auctioneer, bidResult);
			try
			{
				sendMsg(msg);
			}
			catch (Exception e)
			{
				logger.severe("Error sending message \"" + msg + "\" while bidding.");
				e.printStackTrace();
			}
		}
	}

	/**
	  * Process an external bid related to a certain decision.
	  *
	  * @param	bidder	name of the agent who sent the bid
	  * @param	madem_bid	bid containing bidder's preferences
	  */
	public void processBid(String bidder, Literal madem_bid)
	{
		// Get MADeM decision data
		int id_decision = (int)((NumberTermImpl) madem_bid.getTerm(0)).solve();
		DecisionData decisionData = mademData.get(new Integer(id_decision));
		if (decisionData != null)
		{
		    // Add utility values to MADeM decision data
			decisionData.addBid(bidder, ((ListTermImpl) madem_bid.getTerm(1)).getAsList());
			
			// Call Winner Determination Problem if it is the last bid
			if (decisionData.areAllBidsReceived())
				winnerDeterminationProblem(id_decision);
		}
	}

	/**
	  * Obtains the utility value given by a specific utility function.
	  *
	  * @param	auctioneer	agent asking for preference
	  * @param	allocation	allocation being evaluated
	  * @param	uFunc	name of the utility function being considered
	  * @return	the utility value assigned to the allocation. Utility values are
	  *         between [0, 1]; 0 meaning no preference and 1 meaning the highest preference 
	  */
	public float getUtility(String auctioneer, Allocation allocation, Atom uFunc)
	{
		float utility = 0;
		
		UtilityFunctionInt utilityFunction = utilityFunctions.get(uFunc);
		if (utilityFunction != null) {
			try
			{
				utility = utilityFunction.computeUtility(auctioneer, allocation, this);
			}
			catch (UtilityFunctionException e)
			{
				logger.warning("Error computing utility function for the allocation \"" + allocation + "\".");
				e.printStackTrace();
			}
        } else {
            logger.warning("Utility function \"" + uFunc + "\" can not be computed. It seems not to be registered.");
        }
		
		return utility;
	}

	/**
	  * Solves the winner determination problem for a specific decision.
	  *
	  * @param	id_decision	Identifier of the decision being solved
	  */
	public void winnerDeterminationProblem(int id_decision)
	{
		Vector<Float> uValues = new Vector<Float>();
		DecisionData decision = null;
		int i, j, k;
		
		// Retrieve MADeM decision data
		synchronized (mademData)
		{
			decision = mademData.remove(id_decision);
		}

		// Avoid concurrency when timeout concurs with the reception of the last bid
		if (decision == null)
			return;

		// Get MADeM decision data
		logger.severe("Starting Winner Determination Problem.");
		Vector<Atom> agents = decision.getAgents();
		ConcurrentHashMap<Atom, Float> personalWeights = (ConcurrentHashMap<Atom, Float>) decision.getPersonalWeights();
		Vector<Allocation> allocs = decision.getAllocations();
		Vector<Atom> uFuncs = decision.getUtilityFunctions();
		ConcurrentHashMap<Atom, Float> utilityWeights = (ConcurrentHashMap<Atom, Float>) decision.getUtilityWeights();
		ConcurrentHashMap<Atom, Vector<Float>> utilityValues = (ConcurrentHashMap<Atom, Vector<Float>>) decision.getUtilityValues();
		Cuf welfare = decision.getWelfare();

		logger.fine("Agents: " + agents);
		logger.fine("Personal Weights: " + personalWeights);
		logger.fine("Allocations: " + allocs);
		logger.fine("Utility Functions: " + uFuncs);
		logger.fine("Utility Weights: " + utilityWeights);
		logger.fine("Welfare: " + welfare);
		logger.fine("Utility Values (agent = [Allocations X UtilityFunctions]):\n" + utilityValues);
		
		// Weight preferences
		Iterator<Map.Entry<Atom, Vector<Float>>> it = utilityValues.entrySet().iterator();
		Map.Entry<Atom, Vector<Float>> element;
		Float w;
		while (it.hasNext())
		{
			element = it.next();
			uValues = element.getValue();
			w = personalWeights.get(element.getKey());
			if (w != null)
			{
				for (i = 0; i < uValues.size(); i++)
				{
					uValues.set(i, w * uValues.get(i));
				}
			}
		}
		logger.fine("Weighted Utility Values (agent = [Allocations X UtilityFunctions*PersonalWeights]):\n" + utilityValues);
		
		// Winner Determination Problem for each utility function
		float[][] utilityWinners = new float[uFuncs.size()][3]; //Dimensions mean [uFunc_idx][alloc_idx, welfare]
		float utilityValue, socialWelfare;
		Vector<Float> agentBids = null;
		for (i = 0; i < uFuncs.size(); i++)
		{
			// Set initial winner to none
			utilityWinners[i][0] = utilityWinners[i][1] = -1;
			
			for (j = 0; j < allocs.size(); j++)
			{
				socialWelfare = utilityValues.get(agents.get(0)).get(j*uFuncs.size() + i);
				for (k = 1; k < agents.size(); k++)
				{		
					agentBids = utilityValues.get(agents.get(k)); 
					if (agentBids != null) {
						utilityValue = agentBids.get(j*uFuncs.size() + i);
					}
					else { // The agent did not bid on time
						utilityValue = 0;
					}

					switch (welfare)
					{
						case UTILITARIAN:
						socialWelfare += utilityValue;
						break;
	
						case EGALITARIAN:
						if (socialWelfare > utilityValue)
							socialWelfare = utilityValue;
						break;
	
						case ELITIST:
						if (socialWelfare < utilityValue)
							socialWelfare = utilityValue;
						break;
	
						case NASH:
						socialWelfare *= utilityValue;
						break;
	
						default:
						break;
					}
				}
				
				if (socialWelfare >= utilityWinners[i][1])
				{
					utilityWinners[i][0] = j;
					utilityWinners[i][1] = socialWelfare;
				}
			}
			logger.fine("Winner of utility " + uFuncs.get(i) + ": [" + allocs.get((int)(utilityWinners[i][0])) +
						  ", " + utilityWinners[i][1] + "]");
		}
		
		// Multi-modal decision making
		float[] mademWinner = new float[4];  //Elements are [alloc_idx, sw, uFunc_idx]
		mademWinner[0] = mademWinner[1] = mademWinner[2] = -1;
		for (i = 0; i < uFuncs.size(); i++)
		{
			w = utilityWeights.get(uFuncs.get(i));
			if (w == null)
				socialWelfare = utilityWinners[i][1];
			else
				socialWelfare = utilityWinners[i][1] * w;

			if (socialWelfare >= mademWinner[1])
			{
				mademWinner[0] = utilityWinners[i][0];
				mademWinner[1] = socialWelfare;
				mademWinner[2] = i;
			}
		}
		logger.fine("Madem Winner " + uFuncs.get((int) mademWinner[2]) + ": [" + allocs.get((int)mademWinner[0]) +
					  ", " + mademWinner[1]+ "]");
		
		// Publish winner decision
		Literal l = ASSyntax.createLiteral("madem_result");
		l.addTerm(new NumberTermImpl((double) id_decision));
		if (mademWinner[1] != 0)
			l.addTerm((ListTermImpl) (allocs.get((int)mademWinner[0])));
		else
			l.addTerm(new Atom("none"));
			
		logger.fine("Add believe: " + l);
		try
		{
			getTS().getAg().addBel(l);
		}	
		catch (Exception e)
		{
			logger.warning("Error adding believe " + l);
			e.printStackTrace();
		}
		
		logger.fine("Ending Winner Determination Problem.");
	}

}
