// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://grev.uv.es
// http://www.uv.es/agentes
//
//----------------------------------------------------------------------------

package jmadem;

import jason.asSemantics.DefaultInternalAction;
import jason.asSemantics.TransitionSystem;
import jason.asSemantics.Unifier;
import jason.asSyntax.Term;
import jason.asSyntax.VarTerm;
import jason.asSyntax.Literal;
import jason.asSyntax.ListTermImpl;
import jason.asSyntax.NumberTermImpl;
import jason.asSyntax.Atom;

import java.util.logging.Logger;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Iterator;
import java.util.List;

/** 
 * Internal function for starting a MADeM decision with all the parameters
 * being specified in the call (i.e. personal weights, utility weights, welfare and timeout).
 * <p>
 * <b>Use:</b> <br/> <tt>
 * <table border="0">
 *   <tr><td>jmadem.launch_specific_decision(</td><td>[AgentNames],</td><td></td></tr> 
 *   <tr><td></td><td>[Allocations],</td><td>// Obtained through jmadem.construct_allocation</td></tr> 
 *   <tr><td></td><td>[UtilityFunctionNames],</td><td></td></tr> 
 *   <tr><td></td><td>[ [AgentName, Weight] | ListOfPairs ],</td><td></td></tr> 
 *   <tr><td></td><td>[ [UtilityName, Weight] | ListOfPairs ],</td><td></td></tr> 
 *   <tr><td></td><td>Welfare,</td><td></td></tr> 
 *   <tr><td></td><td>Timeout,</td><td></td></tr> 
 *   <tr><td></td><td>DecisionId)</td><td>// Output parameter</td></tr> 
 * </table></tt>
 * </p>
 * <p>
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. 
 * </p>
 * <p>
 * Once MADeM has been resolved, a belief of the type <tt>+madem_result(DecisionId, WinnerAllocation)</tt> 
 * will be added to the agent's belief base.
 * </p>
 *
 * @author Francisco Grimaldo
 */
public class launch_specific_decision extends DefaultInternalAction  {

	private static final long serialVersionUID = -5768570347409861849L;
	
	Logger logger = Logger.getLogger(launch_specific_decision.class.getName());

	@Override
	public Object execute(TransitionSystem ts, Unifier un, Term[] args) throws Exception 
	{
		try
		{
			// Read the parameters (Agents, Allocations, Utility Functions, 
			//                      Personal Weights, Utility Weights, Welfare, Tout)
			Term agentsTerm = args[0];
			if (agentsTerm instanceof VarTerm)
				agentsTerm = ((VarTerm) agentsTerm).getValue();
			ListTermImpl ags = (ListTermImpl) agentsTerm;
			Term allocsTerm = args[1];
			if (allocsTerm instanceof VarTerm)
				allocsTerm = ((VarTerm) allocsTerm).getValue();
			ListTermImpl allocs = (ListTermImpl) allocsTerm;
			Term uFuncsTerm = args[2];
			if (uFuncsTerm instanceof VarTerm)
				uFuncsTerm = ((VarTerm) uFuncsTerm).getValue();
			ListTermImpl uFuncs = (ListTermImpl) uFuncsTerm;
			ConcurrentMap<Atom, Float> personalWeights = new ConcurrentHashMap<Atom, Float>();
			ConcurrentMap<Atom, Float> utilityWeights = new ConcurrentHashMap<Atom, Float>();
			Term pwTerm = args[3];
			if (pwTerm instanceof VarTerm)
				pwTerm = ((VarTerm) pwTerm).getValue();
			Iterator<Term> it = ((ListTermImpl) pwTerm).iterator();
			while (it.hasNext())
			{
				List<Term> pair = ((ListTermImpl) it.next()).getAsList();
				personalWeights.put( (Atom) (pair.get(0)), 
								   new Float( ((NumberTermImpl) pair.get(1)).solve()) );
			}
			Term uwTerm = args[4];
			if (uwTerm instanceof VarTerm)
				uwTerm = ((VarTerm) uwTerm).getValue();
			it = ((ListTermImpl) uwTerm).iterator();
			while (it.hasNext())
			{
				List<Term> pair = ((ListTermImpl) it.next()).getAsList();
				utilityWeights.put( (Atom) (pair.get(0)), 
								   new Float( ((NumberTermImpl) pair.get(1)).solve()) );
			}
			Term welfareTerm = args[5];
			if (welfareTerm instanceof VarTerm)
				welfareTerm = ((VarTerm) welfareTerm).getValue();
			String name_welfare = ((Literal) welfareTerm).toString().toUpperCase();
			Cuf w = (Cuf)Enum.valueOf(Cuf.class, name_welfare);
			Term toutTerm = args[6];
			if (toutTerm instanceof VarTerm)
				toutTerm = ((VarTerm) toutTerm).getValue();
			int tout = Float.valueOf( (float) ((NumberTermImpl) toutTerm).solve() ).intValue();

			// Check whether parameters are correct
			if (allocs.size() == 0)
			{
				logger.severe("Error reading parameters: Allocations cannot be empty.");
				return false;
			}
			if (uFuncs.size() == 0)
			{
				logger.severe("Error reading parameters: Utility Functions cannot be empty.");
				return false;
			}
			if (ags.size() != personalWeights.size())
			{
				logger.severe("Error reading parameters: Agents and Personal Weights mismatch.");
				return false;
			}
			if (uFuncs.size() != utilityWeights.size())
			{
				logger.severe("Error reading parameters: Utility Functions and Utility Weights mismatch.");
				return false;
			}
			
			// Start MADeM decision
			MADeMAgArch agArch = (MADeMAgArch)ts.getUserAgArch();
			int id_decision = agArch.startMADeMDecision(ags, allocs, uFuncs, personalWeights, 
									                    utilityWeights, w, tout);
			if (id_decision == -1)
				return false;
			else
			{
				return un.unifies(args[7], new NumberTermImpl((double) id_decision));
			}
		}
		catch (Exception e)
		{
			logger.severe("Error reading parameters for starting MADeM.");
			e.printStackTrace();
			return false;
		}
    }
}

