// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://www.uv.es/grimo
// http://grev.uv.es
//
//----------------------------------------------------------------------------

/** 
 * J-MADeM Plan Library:
 * ---------------------
 * This file contains a set of plans that are useful for invoking the MADeM 
 * decision making from your AgentSpeak code without paying much attention to 
 * the implementation details of the J-MADeM Internal Actions Library.
 */

/**
 * Plan for setting the social welfare of the MADeM agent.
 * 
 * Use: Add the belief jmadem_welfare(Welfare)
 *      where Welfare = utilitarian, egalitarian, elitist, nash  (Atom format)
 */
+jmadem_welfare(Welfare)
  <- jmadem.set_welfare(Welfare).

/**
 * Plan for reseting the social welfare of the MADeM agent.
 * 
 * Use: Remove the belief jmadem_welfare(_) to reset the welfare to utilitarian
 */
-jmadem_welfare(_)
  <- jmadem.set_welfare(elitist).
  
  
/**
 * Plan for setting the type of multimodality used by the MADeM agent.
 * 
 * Use: Add the belief jmadem_multimodality(Multimodality)
 *      where Multimodality = competitive, aggregate  (Atom format)
 */
+jmadem_multimodality(Multimodality)
  <- jmadem.set_multimodality(Multimodality).

/**
 * Plan for reseting the type of multimodality used by the MADeM agent.
 * 
 * Use: Remove the belief jmadem_multimodality(_) to reset the multimodality to competitive
 */
-jmadem_multimodality(_)
  <- jmadem.set_multimodality(competitive).
  

/**
 * Plans for setting the bid timeout (in ms) in the MADeM agent.
 * 
 * Use: Add the belief jmadem_timeout(Timeout)
 *      with Timeout being a numerical value in milliseconds
 * 		Remove the belief jmadem_timeout(_) to reset the timeout to 1000 ms
 */
+jmadem_timeout(Timeout)
  <- jmadem.set_timeout(Timeout).

/**
 * Plan for reseting the bid timeout (in ms) in the MADeM agent.
 * 
 * Use: Remove the belief jmadem_timeout(_) to reset the timeout to 1000 ms
 */
-jmadem_timeout(_)
  <- jmadem.set_timeout(1000).


/**
 * Plan for setting a personal weight of the MADeM agent.
 * 
 * Use: Add the belief jmadem_personal_weight(AgentName, Weight)
 *		where AgentName is the name of the agent affected
 *      and Weight is the weight applied to his/her utilities
 */
+jmadem_personal_weight(AgentName,Weight)
	<- jmadem.set_personal_weight(AgentName,Weight).
	
/**
 * Plan for reseting a personal weight of the MADeM agent.
 * 
 * Use: Remove the belief jmadem_personal_weight(AgentName,_)
 *      in order to set to 1 the weight of the AgentName 
 */
-jmadem_personal_weight(AgentName,_)
	<- jmadem.set_personal_weight(AgentName,1).
	
	
/**
 * Plan for setting a list of personal weights for the MADeM agent.
 * 
 * Use: Add the belief jmadem_list_of_personal_weights(PW)
 *      where PW = [ jmadem_personal_weight(AgentName,Weight) | Tail ]
 */
+jmadem_list_of_personal_weights(PW) 
	<- jmadem.set_list_of_personal_weights(PW).
	
/**
 * Plan for reseting a list of personal weights for the MADeM agent.
 * 
 * Use: Add the belief jmadem_list_of_personal_weights(PW)
 *      where PW = [ jmadem_personal_weight(AgentName,_) | Tail ]
 */
-jmadem_list_of_personal_weights(PW) 
	<- jmadem.reset_personal_weights(PW). 


/**
 * Plan for setting an utility weight of the MADeM agent.
 * 
 * Use: Add the belief jmadem_utility_weight(UName,Weight)
 *      where UName is the name of the utility function
 *		and Weight is the weight applied to the utility
 */
+jmadem_utility_weight(UName,Weight)
	<- jmadem.set_utility_weight(UName,Weight).
	
/**
 * Plan for reseting an utility weight of the MADeM agent.
 * 
 * Use: Remove the belief jmadem_utility_weight(UName,_)
 *      in order to set to 1 the weight of the utility UName 
 */
-jmadem_utility_weight(UName,_)
	<- jmadem.set_utility_weight(UName,1).	 

	
/**
 * Plan for setting a list of utility weights for the MADeM agent.
 * 
 * Use: Add the belief jmadem_list_of_utility_weights(UW)
 *      where UW = [ jmadem_utility_weight(UName,Weight) | Tail ]
 */	
+jmadem_list_of_utility_weights(UW) 
	<- jmadem.set_list_of_utility_weights(UW).
	
/**
 * Plan for reseting a list of utility weights for the MADeM agent.
 * 
 * Use: Remove the belief jmadem_list_of_utility_weights(UW)
 *      where UW = [ jmadem_utility_weight(UName,_) | Tail ]
 */	
-jmadem_list_of_utility_weights(UW) 
	<- jmadem.reset_utility_weights(UW).
	 

/**
 * Plan for adding a Java-based utility function to the MADeM agent.
 * 
 * Use: Add the belief jmadem_utility_function(UFName, UFClass)
 *      with UFName being the name of the utility function (Atom format)
 *      and UFClass the fully qualified name of the utility function Java class (String format)
 */
+jmadem_utility_function(UFName, UFClass) 
  <- jmadem.add_utility_function(UFName, UFClass).

/**
 * Plan for removing a Java-based utility function to the MADeM agent.
 * 
 * Use: Remove the belief jmadem_utility_function(UFName, UFClass)
 *      with UFName being the name of the utility function (Atom format)
 *      and UFClass the fully qualified name of the utility function Java class (String format)
 */
-jmadem_utility_function(UFName,_)
  <- jmadem.remove_utility_function(UFName).

/**
 * Plans for getting the names of the Java-based utility functions registered for the MADeM agent.
 * 
 * Use: Add the goal jmadem_get_utility_function_names(UtilityNames)
 *      with UtilityNames being the list of utility function names (List of Atoms format)
 */
+!jmadem_get_utility_function_names(UtilityNames)
  <- jmadem.get_utility_function_names(UtilityNames).
-!jmadem_get_utility_function_names(UtilityNames)
  <- .println("Failed getting MADeM utility function names for:");
  	 .println("	UtilityNames = ", UtilityNames). 
	 	 

/**
 * Plan for adding a Java-based filter function to the MADeM agent.
 * 
 * Use: Add the belief jmadem_filter_function(FName, FClass)
 *      with FName being the name of the filter function (Atom format) 
 *      and FClass being the fully qualified name of the filter function Java class (String format)
 */
+jmadem_filter_function(FName, FClass)
  <- jmadem.add_filter_function(FName, FClass).
  
/**
 * Plan for removing a Java-based filter function to the MADeM agent.
 * 
 * Use: Remove the belief jmadem_filter_function(FName, FClass)
 *      with FName being the name of the filter function (Atom format) 
 *      and FClass being the fully qualified name of the filter function Java class (String format)
 */
-jmadem_filter_function(FName,_)
  <- jmadem.remove_filter_function(FName).
  
/**
 * Plans for getting the names of the Java-based filter functions registered for the MADeM agent.
 * 
 * Use: Add the goal jmadem_get_filter_function_names(FilterNames)
 *      with FilterNames being the list of filter function names (List of Atoms format)
 */
+!jmadem_get_filter_function_names(FilterNames)
  <- jmadem.get_filter_function_names(FilterNames).
-!jmadem_get_filter_function_names(FilterNames)
  <- .println("Failed getting MADeM filter function names for:");
  	 .println("	FilterNames = ", FilterNames). 


/**
 * Plans for constructing a set of allocations by instantiating task slots to element values.
 * 
 * Use: Add the goal jmadem_construct_allocations(TaskSlots, Elements, Allocations)
 *      with TaskSlots being the task-slots to to be assigned
 *		and Elements being the logic formula selecting the elements to be assigned from the belief base
 *		and Allocations the assignments being constructed
 *
 * Example: !jmadem_construct_allocations(use(coffeeMachine, Agent), 
 *                                        agent(Agent),
 *										  [use(cofeeMachine, doug), use(cofeeMachine, norman)] )
 *          provided that the belief base contains the literals agent(doug) and agent(norman)
 */
+!jmadem_construct_allocations(TaskSlots, Elements, Allocations)
  : .findall(TaskSlots,Elements,Allocations).
-!jmadem_construct_allocations(TaskSlots, Elements,_)
  <- .println("Failed constructing MADeM allocations for:");
  	 .println("	TaskSlots = ", TaskSlots);
  	 .println("	Elements = ", Elements).

/**
 * Plans for constructing a set of allocations by instantiating task slots to element values.
 * 
 * Use: Add the goal jmadem_construct_allocations(Task, Slots, Elements, Allocations)
 *      with Task being the task with all the parameters involved (Literal format)
 *		and Slots being the list of slots to be assigned (List of Atoms format)
 *		and Elements = [ [ElementValues] | ListOfElementValues ] being the elements to be assigned to the Slots
 *		and Allocations the assignments being constructed
 *
 * Example: !jmadem_construct_allocations(use(coffeeMachine, age).
 *									 	  [age],
 *										  [ [doug, norman] ],
 *										  [use(cofeeMachine, doug), use(cofeeMachine, norman)] )
 */
+!jmadem_construct_allocations(Task, Slots, Elements, Allocations)
  <- jmadem.construct_allocations(Task, Slots, Elements, Allocations).
-!jmadem_construct_allocations(Task, Slots, Elements,_)
  <- .println("Failed constructing MADeM allocations for:");
  	 .println("	Task = ", Task);
  	 .println("	Slots = ", Slots);
  	 .println("	Elements = ", Elements).

/**
 * Plans for constructing a set of allocations by instantiating task slots to element values, where the 
 * validity of each allocation is checked through a set of filter functions.
 * 
 * Use: Add the goal jmadem_construct_allocations(Task, Slots, Elements, FilterNames, Allocations)
 *      with Task being the task with all the parameters involved (Literal format)
 *		and Slots being the list of slots to be assigned (List of Atoms format)
 *		and Elements = [ [ElementValues] | ListOfElementValues ] being the elements to be assigned to the Slots
 *      and FilterNames = [FilterFunctionName | ListOfNames] being the list of filter function names (List of Atoms format)
 *		and Allocations the assignments being constructed
 *
 * Example: !jmadem_construct_allocations(use(coffeeMachine, age).
 *									 	  [age],
 *										  [ [doug, norman] ],
 *                                        [skipDoug],   
 *										  [use(cofeeMachine, norman)] )
 */
+!jmadem_construct_allocations(Task, Slots, Elements, FilterNames, Allocations)
  <- jmadem.construct_allocations(Task, Slots, Elements, FilterNames, Allocations).
-!jmadem_construct_allocations(Task, Slots, Elements, FilterNames,_)
  <- .println("Failed constructing MADeM allocations for:");
  	 .println("	Task = ", Task);
  	 .println("	Slots = ", Slots);
  	 .println("	FilterNames = ", FilterNames);
  	 .println("	Elements = ", Elements).


/**
 * Plans for filtering a list of allocations, where the validity of each allocation is checked through 
 * a filtering function.
 * 
 * Use: Add the goal jmadem_filter_allocations(FilterName, Allocations, FilteredAllocations)
 *      with FilterName being the name of the filtering function 
 *		and Allocations the list of allocations to filter 
 *		and FilteredAllocations the list of filtered allocations 
 *
 * Example: !jmadem_filter_allocations(skipDoug,
 *                                     [use(cofeeMachine, doug), use(cofeeMachine, norman)],
 *                                     [use(cofeeMachine, norman)] ).
 *          provided that the filtering function skipDoug is defined either by means of the rules:
 *             jmadem_filter(skipDoug, use(_, doug)).
 *             or jmadem_filter(skipDoug, use(_, Agent)) :- Agent == doug.
 *          or through the plans:
 *             +!jmadem_filter(skipDoug, use(_, doug)).
 *             or +!jmadem_filter(skipDoug, use(_, Agent)) : Agent == doug.
 *             or +!jmadem_filter(skipDoug, use(_, Agent)) <- Agent == doug.
 */
+!jmadem_filter_allocations(FilterName, Allocations, FilteredAllocations)
  : not .list(FilterName)
  <- !jmadem_filter_allocations(FilterName, Allocations, [], FilteredAllocations).
+!jmadem_filter_allocations(_,[], FilteredAllocations, FilteredAllocations).
+!jmadem_filter_allocations(FilterName,[Allocation|Allocations], PartiallyFilteredAllocations, FilteredAllocations)
  <- !jmadem_invoke_filter(jmadem_filter(FilterName, Allocation), Res);
  	 if ( Res == true )
     {
     	!jmadem_filter_allocations(FilterName, Allocations, PartiallyFilteredAllocations, FilteredAllocations)
     } else {
     	!jmadem_filter_allocations(FilterName, Allocations, [Allocation | PartiallyFilteredAllocations], FilteredAllocations)
     }.


/**
 * Plans for filtering a list of allocations, where the validity of each allocation is checked through 
 * a set of filtering functions.
 * 
 * Use: Add the goal jmadem_filter_allocations(FilterName, Allocations, FilteredAllocations)
 *      with FilterNames = [FilterFunctionName | ListOfNames] being the list of filtering function names (List of Atoms format)
 *		and Allocations the list of allocations to filter 
 *		and FilteredAllocations the list of filtered allocations 
 *
 * Example: !jmadem_filter_allocations([skipDoug, skipNorman],
 *                                     [use(cofeeMachine, doug), use(cofeeMachine, norman)],
 *                                     [] ).
 *          provided that the filtering functions skipDoug and skipNorman are defined either by means of the rules:
 *             jmadem_filter(skipDoug, use(_, doug)).
 *             or jmadem_filter(skipDoug, use(_, Agent)) :- Agent == doug.
 *             jmadem_filter(skipNorman, use(_, norman)).
 *             or jmadem_filter(skipNorman, use(_, Agent)) :- Agent == norman.
 *          or through the plans:
 *             +!jmadem_filter(skipDoug, use(_, doug)).
 *             or +!jmadem_filter(skipDoug, use(_, Agent)) : Agent == doug.
 *             or +!jmadem_filter(skipDoug, use(_, Agent)) <- Agent == doug.
 *             +!jmadem_filter(skipNorman, use(_, norman)).
 *             or +!jmadem_filter(skipNorman, use(_, Agent)) : Agent == norman.
 *             or +!jmadem_filter(skipNorman, use(_, Agent)) <- Agent == norman.
 */
+!jmadem_filter_allocations(FilterNames, Allocations, FilteredAllocations)
  : .list(FilterNames)
  <- !jmadem_filter_allocations_list(FilterNames, Allocations, [], FilteredAllocations).
+!jmadem_filter_allocations_list(_,[], FilteredAllocations, FilteredAllocations).
+!jmadem_filter_allocations_list(FilterNames,[Allocation|Allocations], PartiallyFilteredAllocations, FilteredAllocations)
  <- !jmadem_filter_list(FilterNames, Allocation, false, Res);
     if ( Res == true )
     {
       !jmadem_filter_allocations_list(FilterNames, Allocations, PartiallyFilteredAllocations, FilteredAllocations)
     } else {
       !jmadem_filter_allocations_list(FilterNames, Allocations, [Allocation | PartiallyFilteredAllocations], FilteredAllocations)
     }.
+!jmadem_filter_list([],_, Res, Res).
+!jmadem_filter_list([FilterName | FilterNames], Allocation, PartialRes, Res)
  <- !jmadem_invoke_filter(jmadem_filter(FilterName, Allocation), FilterRes);
     .eval(Aux, PartialRes | FilterRes);
     !jmadem_filter_list(FilterNames, Allocation, Aux, Res).


/**
 * Plans wrapping the calculation of the filtering function for an allocation so that it can be used 
 * internally by J-MADeM. Thus, this plan should never be invoked by the user.
 *
 * Details: FEvent == jmadem_filter(FilterName,Allocation)
 */
+!jmadem_invoke_filter(FEvent, true) : FEvent. 
+!jmadem_invoke_filter(FEvent, Res) : not FEvent
  <- !FEvent;
     Res = true. 
-!jmadem_invoke_filter(_,false).
// Check registered Java-based filter functions
//-!jmadem_invoke_filter(jmadem_filter(FilterName,Allocation),Res)
//  <- jmadem.doFilter(FilterName,Allocation,Res). 


/** 
 * Plan for starting a MADeM decision with the current internal agent
 * settings (i.e. personal weights, utility weights, welfare and timeout).
 * 
 * Use: Add the goal 
 *      !jmadem_launch_decision( AgentNames,   // List of agent names (List of Atoms format)
 * 							 	 Allocations,  // Obtained through !jmadem_construct_allocation
 *								 UtilityNames, // List of utility function names (List of Atoms format)
 *								 DecisionId)   // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocations) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will return all of them.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocations) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_decision(AgentNames, Allocations, UtilityNames, DecisionId)
  <- jmadem.launch_decision(AgentNames, Allocations, UtilityNames, DecisionId).
-!jmadem_launch_decision(AgentNames, Allocations, UtilityNames,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames).

/** 
 * Plan for starting a MADeM decision with the current internal agent
 * settings (i.e. personal weights, utility weights, welfare and timeout) and getting
 * the welfare of the winner allocation along with the MADeM result.
 * 
 * Use: Add the goal 
 *      !jmadem_launch_decision_welfare( 
 *								 AgentNames,   // List of agent names (List of Atoms format)
 * 							 	 Allocations,  // Obtained through !jmadem_construct_allocation
 *								 UtilityNames, // List of utility function names (List of Atoms format)
 *								 DecisionId)   // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocations, SocialWelfare) 
 *
 * - If there are several winner allocations, this plan will return all of them.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocations, SocialWelfare) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_decision_welfare(AgentNames, Allocations, UtilityNames, DecisionId)
  <- jmadem.launch_decision_welfare(AgentNames, Allocations, UtilityNames, DecisionId).
-!jmadem_launch_decision_welfare(AgentNames, Allocations, UtilityNames,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames).

/** 
 * Plan for starting a MADeM decision with the current internal agent
 * settings (i.e. personal weights, utility weights, welfare and timeout).
 * 
 * Use: Add the goal 
 *      !jmadem_launch_decision1( AgentNames,   // List of agent names (List of Atoms format)
 * 							 	  Allocations,  // Obtained through !jmadem_construct_allocation
 *								  UtilityNames, // List of utility function names (List of Atoms format)
 *								  DecisionId)   // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocation) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will just return one 
 *   of them randomly chosen.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocation) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_decision1(AgentNames, Allocations, UtilityNames, DecisionId)
  <- jmadem.launch_decision1(AgentNames, Allocations, UtilityNames, DecisionId).
-!jmadem_launch_decision1(AgentNames, Allocations, UtilityNames,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames).

/** 
 * Plan for starting a MADeM decision with the current internal agent
 * settings (i.e. personal weights, utility weights, welfare and timeout) and getting
 * the welfare of the winner allocation along with the MADeM result.
 * 
 * Use: Add the goal 
 *      !jmadem_launch_decision1_welfare( 
 * 								 AgentNames,   // List of agent names (List of Atoms format)
 * 							 	 Allocations,  // Obtained through !jmadem_construct_allocation
 *								 UtilityNames, // List of utility function names (List of Atoms format)
 *								 DecisionId)   // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocation, SocialWelfare) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will just return one 
 *   of them randomly chosen.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocation, SocialWelfare) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_decision1_welfare(AgentNames, Allocations, UtilityNames, DecisionId)
  <- jmadem.launch_decision1_welfare(AgentNames, Allocations, UtilityNames, DecisionId).
-!jmadem_launch_decision1_welfare(AgentNames, Allocations, UtilityNames,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames).

/** 
 * Plan for starting a MADeM decision with all the parameters being specified
 * in the call (i.e. personal weights, utility weights, welfare and timeout).
 * 
 * Use: Add the goal 
 *      !jmadem_launch_specific_decision( 
 *								AgentNames,		 // List of agent names (List of Atoms format)
 * 								Allocations,     // Obtained through !jmadem_construct_allocation
 *								UtilityNames,    // List of utility function names (List of Atoms format)
 *								PersonalWeights, // [ jmadem_personal_weight(AgentName,Weight) | Tail ]
 *								UtilityWeights,  // [ jmadem_utility_weight(UName,Weight) | Tail ]
 *								Welfare,		 // utilitarian, egalitarian, elitist or nash
 *                              Multimodality,   // competitive, aggregate
 *								Timeout,		 // Timeout is a numerical value
 *								DecisionId)      // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocations) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will return all of them.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocations) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_specific_decision(AgentNames, Allocations, UtilityNames, PersonalWeights,
								  UtilityWeights, Welfare, Multimodality, Timeout, DecisionId)
  <- jmadem.launch_specific_decision(AgentNames, Allocations, UtilityNames, PersonalWeights, UtilityWeights, Welfare, Multimodality, Timeout, DecisionId).
-!jmadem_launch_specific_decision(AgentNames, Allocations, UtilityNames, PersonalWeights,
								  UtilityWeights, Welfare, Multimodality, Timeout,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames);
  	 .println("	PersonalWeights = ", PersonalWeights);
  	 .println("	UtilityWeights = ", UtilityWeights);
  	 .println("	Welfare = ", Welfare);
  	 .println("	Multimodality = ", Multimodality);
  	 .println("	Timeout = ", Timeout).
  	 
/** 
 * Plan for starting a MADeM decision with all the parameters being specified
 * in the call (i.e. personal weights, utility weights, welfare and timeout) and getting
 * the welfare of the winner allocation along with the MADeM result.
 * 
 * Use: Add the goal 
 *      !jmadem_launch_specific_decision_welfare( 
 * 								AgentNames,		 // List of agent names (List of Atoms format)
 * 								Allocations,     // Obtained through !jmadem_construct_allocation
 *								UtilityNames,    // List of utility function names (List of Atoms format)
 *								PersonalWeights, // [ jmadem_personal_weight(AgentName,Weight) | Tail ]
 *								UtilityWeights,  // [ jmadem_utility_weight(UName,Weight) | Tail ]
 *								Welfare,		 // utilitarian, egalitarian, elitist or nash
 *                              Multimodality,   // competitive, aggregate
 *								Timeout,		 // Timeout is a numerical value
 *								DecisionId)      // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocations, SocialWelfare) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will return all of them.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocations, SocialWelfare) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout]  
 */
+!jmadem_launch_specific_decision_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights,
								          UtilityWeights, Welfare, Multimodality, Timeout, DecisionId)
  <- jmadem.launch_specific_decision_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights, UtilityWeights, Welfare, Multimodality, Timeout, DecisionId).
-!jmadem_launch_specific_decision_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights,
								          UtilityWeights, Welfare, Multimodality, Timeout,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames);
  	 .println("	PersonalWeights = ", PersonalWeights);
  	 .println("	UtilityWeights = ", UtilityWeights);
  	 .println("	Welfare = ", Welfare);
  	 .println("	Multimodality = ", Multimodality);
  	 .println("	Timeout = ", Timeout).
  	 
/** 
 * Plan for starting a MADeM decision with all the parameters being specified
 * in the call (i.e. personal weights, utility weights, welfare and timeout).
 * 
 * Use: Add the goal 
 *      !jmadem_launch_specific_decision1( 
 * 								AgentNames,		 // List of agent names (List of Atoms format)
 * 								Allocations,     // Obtained through !jmadem_construct_allocation
 *								UtilityNames,    // List of utility function names (List of Atoms format)
 *								PersonalWeights, // [ jmadem_personal_weight(AgentName,Weight) | Tail ]
 *								UtilityWeights,  // [ jmadem_utility_weight(UName,Weight) | Tail ]
 *								Welfare,		 // utilitarian, egalitarian, elitist or nash
 *                              Multimodality,   // competitive, aggregate
 *								Timeout,		 // Timeout is a numerical value
 *								DecisionId)      // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocation) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will just return one 
 *   of them randomly chosen.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocation) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_specific_decision1(AgentNames, Allocations, UtilityNames, PersonalWeights,
								   UtilityWeights, Welfare, Multimodality, Timeout, DecisionId)
  <- jmadem.launch_specific_decision1(AgentNames, Allocations, UtilityNames, PersonalWeights, UtilityWeights, Welfare, Multimodality, Timeout, DecisionId).
-!jmadem_launch_specific_decision1(AgentNames, Allocations, UtilityNames, PersonalWeights,
								   UtilityWeights, Welfare, Multimodality, Timeout,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames);
  	 .println("	PersonalWeights = ", PersonalWeights);
  	 .println("	UtilityWeights = ", UtilityWeights);
  	 .println("	Welfare = ", Welfare);
  	 .println("	Multimodality = ", Multimodality);
  	 .println("	Timeout = ", Timeout).

/** 
 * Plan for starting a MADeM decision with all the parameters being specified
 * in the call (i.e. personal weights, utility weights, welfare and timeout) and getting
 * the welfare of the winner allocation along with the MADeM result.
 * 
 * Use: Add the goal 
 *      !jmadem_launch_specific_decision1_welfare( 
 * 								AgentNames,		 // List of agent names (List of Atoms format)
 * 								Allocations,     // Obtained through !madem_construct_allocation
 *								UtilityNames,    // List of utility function names (List of Atoms format)
 *								PersonalWeights, // [ jmadem_personal_weight(AgentName,Weight) | Tail ]
 *								UtilityWeights,  // [ jmadem_utility_weight(UName,Weight) | Tail ]
 *								Welfare,		 // utilitarian, egalitarian, elitist or nash
 *                              Multimodality,   // competitive, aggregate
 *								Timeout,		 // Timeout is a numerical value
 *								DecisionId)      // Output parameter
 *
 * Details:
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. Once MADeM has been 
 * resolved, a belief will be added to the agent's belief base of the type:
 * 				 +jmadem_result(DecisionId, WinnerAllocation, SocialWelfare) 
 *
 * It must be noticed that,
 * - If there are several winner allocations, this plan will just return one 
 *   of them randomly chosen.
 *
 * - If no winner allocation can be found the result belief will be of the type:
 * 				 +jmadem_result(DecisionId, none) 
 *
 * - If timeout is reached the result belief will be added with the annotation 'timeout', such as:
 * 				 +jmadem_result(DecisionId, WinnerAllocation, SocialWelfare) [timeout] or,
 * 				 +jmadem_result(DecisionId, none) [timeout] 
 */
+!jmadem_launch_specific_decision1_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights,
								           UtilityWeights, Welfare, Multimodality, Timeout, DecisionId)
  <- jmadem.launch_specific_decision1_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights, UtilityWeights, Welfare, Multimodality, Timeout, DecisionId).
-!jmadem_launch_specific_decision1_welfare(AgentNames, Allocations, UtilityNames, PersonalWeights,
								           UtilityWeights, Welfare, Multimodality, Timeout,_) 
  <- .println("Failed MADeM invokation for:");
  	 .println("	AgentNames = ", AgentNames);
  	 .println("	Allocations = ", Allocations);
  	 .println("	UtilityNames = ", UtilityNames);
  	 .println("	PersonalWeights = ", PersonalWeights);
  	 .println("	UtilityWeights = ", UtilityWeights);
  	 .println("	Welfare = ", Welfare);
  	 .println("	Multimodality = ", Multimodality);
  	 .println("	Timeout = ", Timeout).

  	 
/**
 * Plans wrapping the calculation of the utility value of an allocation so that it can be used 
 * internally by J-MADeM. Thus, this plan should never be invoked by the user.
 *
 * Details: Id == [auctioneer,mademAuctionId,utilityId]
 *          UFEvent == jmadem_utility(UFName,Auctioneer,Allocation,Utility)
 */
@jmadem_compute_utility_rule[atomic]
+!jmadem_compute_utility(Id, jmadem_utility(UFName,Auctioneer,Allocation,Utility)) : jmadem_utility(UFName,Auctioneer,Allocation,Utility)
  <- +jmadem_compute_utility(Id, Utility). 
 
@jmadem_compute_utility_plan[atomic]
+!jmadem_compute_utility(Id, jmadem_utility(UFName,Auctioneer,Allocation,Utility)) : not jmadem_utility(UFName,Auctioneer,Allocation,Utility)
  <- !jmadem_utility(UFName,Auctioneer,Allocation,Utility);
     +jmadem_compute_utility(Id, Utility). 
  