/**
 * habitant agent
 *
 * @author Fco. Javier Cordero Cadenas, Francisco Grimaldo, Fernando Barber
 */

{ include("../../asl/jmadem.asl") }      // J-MADeM Plan Library 

/*
  Facts introduced by the world
*/
// cityVehicle(VehicleId, VehicleType, VehicleSeats, VehicleMaxVel, Consumption)
// myVehicle(VehicleId, VehicleType, VehicleSeats, VehicleMaxVel, Consumption)
// otherVehicle(Owner, VehicleId, VehicleType, VehicleSeats, VehicleMaxVel, Consumption)

/*
  Important facts created by the agent
*/
// Table of time and consumption by vehicle and number of partners
// transport( Owner, VehicleId, VehicleType, VehicleSeats, Time, Consumption, IdealTime, IdealConsumption, Partners)
// Exit time of the agent from home
// exitTime(Time)	

/*
  Rules
*/
jmadem_multimodality(aggregate). //competitive, aggregate

agent(Ag):- neighbor(Ag).
agent(Ag):- .my_name(Ag).

// It is important to indicate capacity in "transport" facts to avoid multiple solutions
vehicle(Veh):-
	transport(_, Veh,_,_,_,_,_,_,1).

owns(Ag, Veh):-
	transport(Ag, Veh,_,_,_,_,_,_,1).

capacity(Veh, Seats):-
	transport(_, Veh,_,Seats,_,_,_,_,1).

capacityOk([]).
capacityOk([usersIn(Vehicle,NumberAg)|Tail]) :-
	capacity(Vehicle,Cap) &
	Cap >= NumberAg & 
	capacityOk(Tail).

numUsers(V,Alloc,N) :-
	.setof(U,.member(travelBy(U,V),Alloc),Users) &
	.length(Users,N).

ownerInOk([],_).
ownerInOk([travelBy(_,V)|Tail],Alloc) :-
	owns(A,V) &
	(A == city | .member(travelBy(A,V),Alloc) ) &
	ownerInOk(Tail,Alloc).

trainTicket(1.0).

/*plans*/

/*
  Agent Initialization
*/

@initAg_plan[atomic]
+initAg : atWork(AW) & myPosition(XAg,YAg) & chargeDay(Cd) & capital(C) & salary(Salary) &
		priceXLitreGas(Price) & kgCo2XLitreGas(Co2Litre) & lowerLevelLearningFactor(LowLearn) & higherLevelLearningFactor(HighLearn) &
		lowerLevelForgettingFactor(LowForget) & higherLevelForgettingFactor(HighForget) & roadLength(RoadLength) &
		roadSpeed(RoadSpeed) & roadFlow(RoadFlow) & toll(Toll)
	<-	.my_name(Me);
		+myPosition(XAg,YAg);
		+atWork(Me,AW);
		+capital(C);
		+salary(Salary);
		+chargeDay(Cd);
		+toll(Toll);
		+priceXLitreGas(Price);
		+kgCo2XLitreGas(Co2Litre);
		+maxTimeOld(2000);
		+maxConsumptionOld(4);
		+lowerLevelLearningFactor(LowLearn);
		+higherLevelLearningFactor(HighLearn);
		+lowerLevelForgettingFactor(LowForget);
		+higherLevelForgettingFactor(HighForget);
		+roadLength(RoadLength);
		+roadSpeed(RoadSpeed);
		+roadFlow(RoadFlow);
		if (groupLeader(Me)[source(percept)]){
			+groupLeader(Me)
		};
		+distCity(Me,RoadLength);
		if (neighbor(_)){
			for(neighbor(N)){
				+neighbor(N)
			}
		};
		.count(neighbor(_)[source(self)],GroupSize);
		+groupSize(GroupSize+1);
		// Cautious Factor. Habitant wants to be at work earlier
		.random(CFactor);
		+cautiousFactor(CFactor);
		// Delay arriving work in the last iteration, 0 initially
		+delay(0);
		// New allocation has to be made
		+groupChanged;
		+alloc([]);
		
		.findall(vehicle(Me,V,T,S,Vel,Cons),myVehicle(V,T,S,Vel,Cons),LVeh);  
		.findall(vehicle(city, V2,T2,S2,Vel2,Cons2),cityVehicle(V2,T2,S2,Vel2,Cons2),LCityVeh);
		.concat(LVeh, LCityVeh, LVeh2);
		.findall(vehicle(Neig,V3,T3,S3,Vel3,Cons3),otherVehicle(Neig,V3,T3,S3,Vel3,Cons3),LOtherVeh);
		.concat(LVeh2, LOtherVeh, LVehicles);
		!addVehicles(RoadLength,RoadSpeed,LVehicles);
			
        // Ojo ! manejamos costes (negativo), los pesos van al reves (en modo competitive)
		// Configuracion casi individualista: 1, 0.1, 0.1
		// Configuracion egalitarian: 1, 0.5, 0.4 
		+jmadem_utility_weight(tempUF, 1);
        +jmadem_utility_weight(econUF, 0.1); //0.45
        +jmadem_utility_weight(co2UF, 0.1);
        +jmadem_welfare(utilitarian); // utilitarian, elitist, nash, egalitarian
        +jmadem_timeout(50000);
		do(ready).

+!addVehicles(_,_,[]).
+!addVehicles(Dist,RoadMaxSpeed,[vehicle(Owner,V,T,S,Vel,Cons)|LV])
	<-	if (Vel <= RoadMaxSpeed){
			Time = (Dist/Vel)*3600
		} 
		else {
			Time = (Dist/RoadMaxSpeed)*3600
		};
		Consumption = (Dist*Cons)/100;
		if(Owner == city) {  // No partners
			// Extra delay for public transport: 5 min
			Time2 = Time + 5*60;
			+transport(Owner,V,T,S,Time2,Consumption,Time2,Consumption,1);
		}
		else {
			// All partner possibilities
			for(.range(Part,1,S)){
				// Extra delay for collecting passengers: 5 min for extra passenger
				Time2 = Time + (Part-1)*5*60;
				+transport(Owner,V,T,S,Time2,Consumption,Time2,Consumption,Part);
			};
		};
		!addVehicles(Dist,RoadMaxSpeed,LV).

//-------------------------------------------------------------------
/*
  Utility functions
*/
		
/** 
 * Temp utility function: Returns the estimated travel time.
 *    Allocation = [travelBy(ag1,vehicle1), ..., travelBy(agN,vehicleN)]
 */
@tempUF[atomic]
+!jmadem_utility(tempUF, _, Allocation, TimeMoney) : .my_name(MyName)
	<-  .member(travelBy(MyName, Vehicle), Allocation);
        !countPartners(Allocation, Vehicle, Partners);
        .term2string(Vehicle,StrVehicle);
        if ( .substring("car", StrVehicle) )
        {
            ?transport(_,Vehicle,_,_,EstTime,_,_,_,Partners);
			// Each partner takes 5 min to collect.
			// But this is already taken into account in EstTime
			ExtraTime = 0;
        } else {
			// Public transport (train, bus).
            ?transport(_,Vehicle,_,_,EstTime,_,_,_,_);
			ExtraTime = 0;
        }
        ?salary(Salary);
        Time = - (EstTime+ ExtraTime);
		// The utility returns the time estimated for the travel converted to money
		TimeMoney = Time * Salary/21*8*60*60.
        //.println("-->TUtility ", Time, " for ",Allocation).

// Count number of persons traveling in a given vehicle
// countPartners(+ List of travelBy(), + Vehicle, ? Num of persons)
+!countPartners([],_,0).
+!countPartners([travelBy(_,Vehicle)|RestOfTravelBy], Vehicle, Partners)
    <- !countPartners(RestOfTravelBy, Vehicle, P);
       Partners = P + 1.
+!countPartners([travelBy(_,Vehicle2)|RestOfTravelBy],Vehicle, Partners) : Vehicle2 \== Vehicle
    <- !countPartners(RestOfTravelBy, Vehicle, Partners).

/** 
 * Econ utility function: Returns the estimated travel cost.
 *    Allocation = [travelBy(ag1,vehicle1), ..., travelBy(agN,vehicleN)]
 */
@econUF[atomic]
+!jmadem_utility(econUF, _, Allocation, Econ) : .my_name(MyName)
    <-  .member(travelBy(MyName, Vehicle), Allocation);
        !countPartners(Allocation, Vehicle, Partners);
        ?priceXLitreGas(PriceLitre);
        ?transport(_,Vehicle, Type,_,_,_,_,_,_);
		if ((Type==car) | (Type==moto) )
        {
			?transport(_,Vehicle, Type,_,_,Consumption,_,_,Partners);
			Econ2 = Consumption * PriceLitre / Partners;
        };
		if (Type==train)
		{
			// Ticket Price
			?trainTicket(Ticket);
			Econ2 = Ticket;
        };
		if (Type==bike)
		{
			Econ2 = 0;
		};
		?salary(Salary);
		if (Econ2 < Salary/30) {
			Econ = - Econ2;
		} else { 
			Econ = - 2*Econ2;
		}.
        //.println("-->EUtility ", Econ, " for ",Allocation).

/** 
 * co2 utility function: Returns the estimated travel cost.
 *    Allocation = [travelBy(ag1,vehicle1), ..., travelBy(agN,vehicleN)]
 */
@co2UF[atomic]
+!jmadem_utility(co2UF, _, Allocation, Co2Money) : .my_name(MyName)
    <-  .member(travelBy(MyName, Vehicle), Allocation);
        !countPartners(Allocation, Vehicle, Partners);
        //!extractLVehicles(Allocation, LVehicles);
        //.length(LVehicles, NVehicles);
        .term2string(Vehicle,StrVehicle);
        if ( .substring("car", StrVehicle) )
        {
            ?transport(_,Vehicle,_,_,_,Cons,_,_,Partners);
			Consumption = Cons / Partners;
        } else {
			// Public transport
            ?transport(_,Vehicle,_,_,_,Consumption,_,_,_);
        }
        ?kgCo2XLitreGas(Co2Litre);
		// Por arreglar !!
        Co2 = -Consumption * Co2Litre;
		// Conversion KgCO2 - Money: 0.1
		Co2Money = Co2 * 0.1. // Por arreglar !!
        //.println("-->Utility ", Co2, " whith NVehicles ", NVehicles, " for ",Allocation).

// Extracts all vehicles used in present iteration
// extractLVehicles(+ List of travelBy(), ? List of vehicles)
+!extractLVehicles([],[]).    
+!extractLVehicles([travelBy(_,Vehicle)|RestOfTravelBy], NewLVehicles)
    <- !extractLVehicles(RestOfTravelBy, LVehicles);
       .union(LVehicles, [Vehicle], NewLVehicles).

//--------------------------------------------------------------------
/*
  Agent cycle
*/
+start <- !start.

-!start <- .println("Ai mare!!").
+!start : iteration(I) & chargeDay(Cd)
	<-	.abolish(start[source(percept)]);
		.my_name(Me);
		+iteration(I);
		
		if (groupLeader(Me)){
			.findall(Veh,transport(_,Veh,_,_,_,_,_,_,1),LVehicles);
			.findall(AgN,neighbor(AgN),LNeighbors);
			.findall(Ag,agent(Ag),LAg);
			LAg = [A1,A2,A3,A4];

            if(groupChanged) // Create new alloc
			{
				//.println("--- Constructing ", LAg);
				.setof(Alloc,
					(Alloc = [travelBy(A1,V1), travelBy(A2,V2), travelBy(A3,V3), travelBy(A4,V4)] & 
					 vehicle(V1) & vehicle(V2) & vehicle(V3) & vehicle(V4)  & 
					 .setof(usersIn(V,N), // Computes a list of users per vehicle for an allocaction
								  (.member(travelBy(_,V),Alloc) &
								  numUsers(V,Alloc,N)), 
								 UsersPerVehicle)  &
					 capacityOk(UsersPerVehicle)   & // An allocation is ok if all the vehicles in it are used under their capacities
					 ownerInOk(Alloc,Alloc)
					), // An agent drives his vehicle if it is used
					NewAlloc);
				-+alloc(NewAlloc);
				-groupChanged;
			}
			
			?alloc(FilteredAlloc);
            //.println("---> ", FilteredAlloc);

			// Launch J-MADeM
            !jmadem_launch_decision1(LAg, FilteredAlloc, [tempUF], DecisionId)
            //!jmadem_launch_decision1(LAg, FilteredAlloc, [econUF], DecisionId)
            //!jmadem_launch_decision1(LAg, FilteredAlloc, [co2UF], DecisionId)
            //!jmadem_launch_decision1(LAg, FilteredAlloc, [tempUF, econUF, co2UF], DecisionId)
            //!jmadem_launch_decision1(LAg, FilteredAlloc, [tempUF, econUF], DecisionId)
		}.
		

+jmadem_result(Id, L)
	<-	.abolish(iteration(_));
		.my_name(Me);
		if (groupLeader(Me)){
			if (neighbor(_)){
				for (neighbor(N)){
					.send(N,tell,jmadem_result(Id,L))
					}
				}
		};
		// gets the travelBy which contains Me
		.member(travelBy(Me,V),L);
		// gets the number of partners in the vehicle
        .term2string(V,StrV);
        if ( .substring("train", StrV) | .substring("bus", StrV) ) {
			Np = 1;
		} else {
			.findall(Ag, .member(travelBy(Ag,V),L), Lpartners);
			.length(Lpartners, Np);
		}
		+partners(Np);
		!mymadem_result(Id,[travelBy(Me,V)]).
		
+!mymadem_result(Id, [travelBy(Ag,Vehicle)]) : atWork(_,TWork)
	<-	?partners(Part);
		?transport(_,Vehicle,Type,_,T,C,_,_,Part);
		.abolish(madem_result(_,_));
		TE = TWork - T ;
		?cautiousFactor(CFactor);
		if ((Type==car) | (Type==moto) | (Type==bike)){
			// wants to be at work between 0..2 min earlier
			TExit = TE - 2*60*CFactor 
		};
		if ((Type==bus)|(Type==train)){
			// Also public transport is not waiting for habitant. 
			// He have to be 5 min earlier
			TExit = TE - 2*60*CFactor // - 5*60 
		};
		+exitTime(TExit);
		travelBy(Vehicle,TWork,TExit,C).
		
//------------------------------------------------------------------
/*
  Calculation of new values.
  Called from World after traffic simulation
*/
//@realValues_plan[atomic]
+realValues(V,T,NewTime,NewCons) : transport(M,V,T,S,OldTime,OldCons,_,_,Part) & capital(Salary) & priceXLitreGas(P) & partners(Part) 
		& maxTime(MTP) & maxConsumption(MCP) & lowerLevelLearningFactor(LowLearn)
		& higherLevelLearningFactor(HighLearn) & lowerLevelForgettingFactor(LowForget) & higherLevelForgettingFactor(HighForget)
		& exitTime(Texit) & atWork(_,Twork)
	<-	.random(W);
		LearnFactor = W*(HighLearn-LowLearn)+LowLearn;
		.random(W2);
		ForgetFactor = W2*(HighForget-LowForget)+LowForget;
		CalculatedTime = NewTime*LearnFactor + (1-LearnFactor)*OldTime;
		CalculatedConsumption = NewCons*LearnFactor + (1-LearnFactor)*OldCons;
		
		//Delay arriving work. - before, + after
		.abolish(delay(_));
		NewDelay = Texit + NewTime - Twork;
		+delay(NewDelay);
		//.println(" Delay -> ", NewDelay);
		
		for (transport(Ow,Veh,T,Se,CT,CC,IdealT,IdealC,Part2)){ 
			.abolish(transport(_,Veh,_,_,_,_,_,_,Part2));
			if(Part == Part2) {
				+transport(Ow,Veh,T,Se,CalculatedTime,CalculatedConsumption,IdealT,IdealC,Part)
			}
			else { // Vehicles not used in current iteration are affected by forget
				CalculatedTime2 = (IdealT*ForgetFactor + (1-ForgetFactor)*CT);
				CalculatedConsumption2 = (IdealC*ForgetFactor + (1-ForgetFactor)*CC);
				+transport(Ow,Veh,T,Se,CalculatedTime2,CalculatedConsumption2,IdealT,IdealC,Part2)
			}
		};

		.abolish(partners(_));
		realValuesUpdated(0).
		
+end
	<-	end(0).
