// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://www.uv.es/grimo
// http://grev.uv.es
//
//----------------------------------------------------------------------------

package jmadem;

import jason.RevisionFailedException;
import jason.asSemantics.Agent;
import jason.asSemantics.Intention;
import jason.asSyntax.ListTerm;
import jason.asSyntax.Literal;
import jason.asSyntax.Term;

import java.util.List;
import java.util.logging.Logger;



/**
  * Agent class for a Jason agent performing MADeM decisions.
  *
  * @author Francisco Grimaldo
  */
public class MADeMAgent extends Agent
{
	/** Class logger */
	private Logger logger = Logger.getLogger(MADeMAgent.class.getName());
	
	
	/**
	  * MADeM Belief Revision Function:  
	  * This function detects those literals coming from an utility function that
	  * has been implemented as an AgentSpeak plan. These literals will be of the 
	  * form <tt>jmadem_compute_utility(Id, U)</tt>, where <tt>Id</tt> is the unique 
	  * identifier of utility computation and <tt>U</tt> is the utility value 
	  * obtained. When detected, the agent catches the utility value identified by 
	  * <tt>Id</tt>. Otherwise, it invokes the usual Belief Revision Function. That
	  * is, it revises the belief base with the given literal to add, to remove, and
	  * the current intention that triggered the operation.
      * 
	  * @param	beliefToAdd	belief to be added to the belief base
	  * @param	beliefToDel	agent class parameters
	  * @param	i			AgentSpeak source code
	  * @return List[0] has the list of actual additions to the belief base,
	  * 		and List[1] has the list of actual deletions; this is used to 
	  * 		generate the appropriate internal events. If nothing change, 
	  * 		returns null.
	  * @throws	RevisionFailedException
	  */
    @Override
    public List<Literal>[] brf(Literal beliefToAdd, Literal beliefToDel,  Intention i) throws RevisionFailedException {

    	if ( (beliefToAdd != null) && (beliefToAdd.getFunctor().equals("jmadem_compute_utility")) )
    	{
    		if (beliefToAdd.getArity() != 2)
    		{
    			logger.warning("Bad formed MADeM utility value belief. Discarding beliefe: " + beliefToAdd);
    			return null;
    		}
    		
    		ListTerm utilityValueId = (ListTerm) beliefToAdd.getTerm(0);
    		Term utilityValue = beliefToAdd.getTerm(1);
    		    		
    		MADeMAgArch agArch = (MADeMAgArch)this.getTS().getUserAgArch();
    		agArch.catchUtilityValue(utilityValueId, utilityValue);
    		
    		return null;
    	}
    	else
    		return super.brf(beliefToAdd, beliefToDel, i);
    }

}
