// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://www.uv.es/grimo
// http://grev.uv.es
//
//----------------------------------------------------------------------------

package jmadem;

import jason.asSemantics.DefaultInternalAction;
import jason.asSemantics.TransitionSystem;
import jason.asSemantics.Unifier;
import jason.asSyntax.ASSyntax;
import jason.asSyntax.ListTerm;
import jason.asSyntax.Literal;
import jason.asSyntax.Term;
import jason.asSyntax.ListTermImpl;

import java.util.logging.Logger;
import java.util.Iterator;
import java.util.List;

/** 
 * Internal function for constructing a set of allocations by instantiating
 * task slots to element values.
 * <p>
 * <b>Use:</b> <br/> <tt>
 * <table border="0">
 *   <tr><td>jmadem.construct_allocations(</td><td>TaskWithAllParameters,</td><td></td></tr> 
 *   <tr><td></td><td>[Slots],</td><td></td></tr> 
 *   <tr><td></td><td>[ [ElementValues] | ListOfElementValues ],</td><td></td></tr> 
 *   <tr><td></td><td>AllocationsOut )</td><td></td></tr> 
 * </table></tt>
 * </p>
 * <p>
 * <b>Example:</b> <br/> <tt>
 * <table border="0">
 *   <tr><td>jmadem.construct_allocations(</td><td>use(coffeeMachine, age),</td><td></td></tr> 
 *   <tr><td></td><td>[age],</td><td></td></tr> 
 *   <tr><td></td><td>[ [doug, norman] ],</td><td></td></tr> 
 *   <tr><td></td><td>[use(cofeeMachine, doug), use(cofeeMachine, norman)] )</td><td></td></tr> 
 * </table></tt>
 * </p>
 *
 * @author Francisco Grimaldo
 */
public class construct_allocations extends DefaultInternalAction  {

	private static final long serialVersionUID = 1714994035302977795L;

	Logger logger = Logger.getLogger(construct_allocations.class.getName());

	Literal task;
	Literal instantiated_task;
	List<Term> params, slots, elements;
	ListTerm allocs = new ListTermImpl();
	ListTerm fFuncs = null;
	MADeMAgArch agArch;

	@Override
	public Object execute(TransitionSystem ts, Unifier un, Term[] args) throws Exception 
	{
		try
		{
		    // Read input parameters (TaskWithAllParameters, [Slots], [ListsOfElementValues])
			task = (Literal) args[0];
			slots = ((ListTerm) args[1]).getAsList();
			elements = ((ListTerm) args[2]).getAsList();
			if (args.length > 4)
				fFuncs = (ListTerm) args[3];
			agArch = (MADeMAgArch)ts.getUserAgArch();

			// Construct allocations
			instantiated_task = ASSyntax.createLiteral(task.getFunctor()); 
			allocs =  new ListTermImpl();			
			instantiate_task(0, 0);

			// Return allocations
			if (args.length > 4)
				return un.unifies(args[4], allocs);
			else
				return un.unifies(args[3], allocs);
		}
		catch (Exception e)
		{
			logger.severe("Error constructing allocations.");
			e.printStackTrace();
			return false;
		}
    }
	
	/**
	  * Constructs the set of allocations by instantiating the slots with the
	  * elements passed to the internal action.
	  *
	  * @param	id_params	index over the parameters of the task
	  * @param	id_slots	index over the slots being considered
	 */
	void instantiate_task(int id_params, int id_slots)
	{
		// Base case
		if (id_params >= task.getArity())
		{
			ListTerm l = new ListTermImpl();
			l.add(instantiated_task.copy());
			if ( (fFuncs == null) || agArch.doFilter(l, fFuncs))
				allocs.add(l);
			return;
		}
		
		// Recursive case
		if ( (id_slots < slots.size()) && (task.getTerm(id_params).equals(slots.get(id_slots))) )
		{
			// Parameter is a slot that can be assigned a set of elements
			instantiated_task.addTerm(task.getTerm(id_params));
			Iterator<Term> it_element = ((ListTerm) elements.get(id_slots)).iterator();
			while (it_element.hasNext())
			{
				instantiated_task.setTerm(id_params, it_element.next());
				instantiate_task(id_params + 1, id_slots + 1);
			}
			instantiated_task.delTerm(id_params);
		}
		else 
		{
			// Parameter is instantiated. No slot available
			instantiated_task.addTerm(task.getTerm(id_params));
			instantiate_task(id_params + 1, id_slots);
			instantiated_task.delTerm(id_params);
		}
	}
}
