// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://www.uv.es/grimo
// http://grev.uv.es
//
//----------------------------------------------------------------------------

package jmadem;

import jason.asSemantics.DefaultInternalAction;
import jason.asSemantics.TransitionSystem;
import jason.asSemantics.Unifier;
import jason.asSyntax.ListTerm;
import jason.asSyntax.Term;
import jason.asSyntax.NumberTermImpl;

import java.util.logging.Logger;

/** 
 * Internal function for starting a MADeM decision with the current internal agent
 * settings (i.e. personal weights, utility weights, welfare and timeout).
 * <p>
 * <b>Use:</b> <br/> <tt>
 * <table border="0">
 *   <tr><td>jmadem.launch_decision(</td><td>[AgentNames],</td><td></td></tr> 
 *   <tr><td></td><td>[Allocations],</td><td>// Obtained through jmadem.construct_allocation</td></tr> 
 *   <tr><td></td><td>[UtilityFunctionNames],</td><td></td></tr> 
 *   <tr><td></td><td>DecisionId)</td><td>// Output parameter</td></tr> 
 * </table></tt>
 * </p>
 * <p>
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. 
 * </p>
 * <p>
 * Once MADeM has been resolved, a belief of the type <tt>+jmadem_result(DecisionId, WinnerAllocations, SocialWelfare)</tt> 
 * will be added to the agent's belief base.
 * </p>
 * <p>
 * It must be noticed that, if there are several winner allocations, this function will return all of them. 
 * </p>
 *
 * @author Francisco Grimaldo
 */
public class launch_decision_welfare extends DefaultInternalAction  {

	private static final long serialVersionUID = -1075003925411067247L;
	
	Logger logger = Logger.getLogger(launch_decision_welfare.class.getName());

	@Override
	public Object execute(TransitionSystem ts, Unifier un, Term[] args) throws Exception 
	{
		try
		{
			// Read the parameters (Agents, Allocations, Utility Functions)
			ListTerm ags = (ListTerm) args[0];
			ListTerm allocs = (ListTerm) args[1];
			ListTerm uFuncs = (ListTerm) args[2];
			
			// Check whether parameters are correct
			if (allocs.size() == 0)
			{
				logger.severe("Error reading parameters: Allocations cannot be empty.");
				return false;
			}
			if (uFuncs.size() == 0)
			{
				logger.severe("Error reading parameters: Utility Functions cannot be empty.");
				return false;
			}

			// Start MADeM decision with the current agent settings
			MADeMAgArch agArch = (MADeMAgArch)ts.getUserAgArch();
			int id_decision = agArch.startMADeMDecision(ags, allocs, uFuncs, true, true);
			if (id_decision == -1)
				return false;
			else
			{
				return un.unifies(args[3], new NumberTermImpl((double) id_decision));
			}
		}
		catch (Exception e)
		{
			logger.severe("Error reading parameters for starting MADeM.");
			e.printStackTrace();
			return false;
		}
    }
}
