// ----------------------------------------------------------------------------
// Copyright (C) 2008 Francisco Grimaldo, Miguel Lozano, Fernando Barber
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
// 
// To contact the authors:
// francisco.grimaldo@uv.es, miguel.lozano@uv.es, fernando.barber@uv.es
// http://www.uv.es/grimo
// http://grev.uv.es
//
//----------------------------------------------------------------------------

package jmadem;

import jason.asSemantics.DefaultInternalAction;
import jason.asSemantics.TransitionSystem;
import jason.asSemantics.Unifier;
import jason.asSyntax.ListTerm;
import jason.asSyntax.NumberTerm;
import jason.asSyntax.Term;
import jason.asSyntax.Literal;
import jason.asSyntax.NumberTermImpl;
import jason.asSyntax.Atom;

import java.util.logging.Logger;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Iterator;

/** 
 * Internal function for starting a MADeM decision with all the parameters
 * being specified in the call (i.e. personal weights, utility weights, welfare and timeout).
 * <p>
 * <b>Use:</b> <br/> <tt>
 * <table border="0">
 *   <tr><td>jmadem.launch_specific_decision_welfare(</td><td>[AgentNames],</td><td></td></tr> 
 *   <tr><td></td><td>[Allocations],</td><td>// Obtained through jmadem.construct_allocation</td></tr> 
 *   <tr><td></td><td>[UtilityFunctionNames],</td><td></td></tr> 
 *   <tr><td></td><td>[ jmadem_personal_weight(AgentName,Weight) | Tail ],</td><td></td></tr> 
 *   <tr><td></td><td>[ jmadem_utility_weight(UName,Weight) | Tail ],</td><td></td></tr> 
 *   <tr><td></td><td>Welfare,</td><td></td></tr> 
 *   <tr><td></td><td>Multimodality,</td><td></td></tr> 
 *   <tr><td></td><td>Timeout,</td><td></td></tr> 
 *   <tr><td></td><td>DecisionId)</td><td>// Output parameter</td></tr> 
 * </table></tt>
 * </p>
 * <p>
 * Launching MADeM is not a blocking task. That is, this functions only starts the 
 * MADeM process and gives the decision identifier as a result. 
 * </p>
 * <p>
 * Once MADeM has been resolved, a belief of the type <tt>+jmadem_result(DecisionId, WinnerAllocations, SocialWelfare)</tt> 
 * will be added to the agent's belief base.
 * </p>
 * <p>
 * It must be noticed that, if there are several winner allocations, this function will return all of them. 
 * </p>
 *
 * @author Francisco Grimaldo
 */
public class launch_specific_decision_welfare extends DefaultInternalAction  {

	private static final long serialVersionUID = -5768570347409861849L;
	
	Logger logger = Logger.getLogger(launch_specific_decision_welfare.class.getName());

	@Override
	public Object execute(TransitionSystem ts, Unifier un, Term[] args) throws Exception 
	{
		try
		{
			// Read the parameters (Agents, Allocations, Utility Functions, 
			//                      Personal Weights, Utility Weights, Welfare, Multimodality, Tout)
			ListTerm ags = (ListTerm) args[0];
			ListTerm allocs = (ListTerm) args[1];
			ListTerm uFuncs = (ListTerm) args[2];
			ConcurrentMap<Atom, Float> personalWeights = new ConcurrentHashMap<Atom, Float>();
			ConcurrentMap<Atom, Float> utilityWeights = new ConcurrentHashMap<Atom, Float>();
			Literal aux;
			Iterator<Term> it = ((ListTerm) args[3]).iterator();
			while (it.hasNext())
			{
				aux = (Literal) it.next();
				personalWeights.put( (Atom) aux.getTerm(0), 
								   new Float( ((NumberTerm) aux.getTerm(1)).solve()) );
			}
			it = ((ListTerm) args[4]).iterator();
			while (it.hasNext())
			{
				aux = (Literal) it.next();
				utilityWeights.put( (Atom) aux.getTerm(0), 
								   new Float( ((NumberTerm) aux.getTerm(1)).solve()) );
			}
			String name_welfare = ((Literal) args[5]).toString().toUpperCase();
			Cuf w = (Cuf)Enum.valueOf(Cuf.class, name_welfare);
			String name_multimodality = ((Literal) args[6]).toString().toUpperCase();
			Multimodality m = (Multimodality)Enum.valueOf(Multimodality.class, name_multimodality);
			int tout = Float.valueOf( (float) ((NumberTerm) args[7]).solve() ).intValue();

			// Check whether parameters are correct
			if (allocs.size() == 0)
			{
				logger.severe("Error reading parameters: Allocations cannot be empty.");
				return false;
			}
			if (uFuncs.size() == 0)
			{
				logger.severe("Error reading parameters: Utility Functions cannot be empty.");
				return false;
			}
			if (ags.size() != personalWeights.size())
			{
				logger.severe("Error reading parameters: Agents and Personal Weights mismatch.");
				return false;
			}
			if (uFuncs.size() != utilityWeights.size())
			{
				logger.severe("Error reading parameters: Utility Functions and Utility Weights mismatch.");
				return false;
			}
			
			// Start MADeM decision
			MADeMAgArch agArch = (MADeMAgArch)ts.getUserAgArch();
			int id_decision = agArch.startMADeMDecision(ags, allocs, uFuncs, personalWeights, 
									                    utilityWeights, w, m, tout, true, true);
			if (id_decision == -1)
				return false;
			else
			{
				return un.unifies(args[8], new NumberTermImpl((double) id_decision));
			}
		}
		catch (Exception e)
		{
			logger.severe("Error reading parameters for starting MADeM.");
			e.printStackTrace();
			return false;
		}
    }
}

