# Script para estimar parámetros g y c del modelo de Gompertz
# Función: lx = lo * g^((c^x) - 1)

# Generar datos sintéticos con ruido
set.seed(123) # Para reproducibilidad

# Parámetros reales (desconocidos en un caso real)
lo_real <- 1000
g_real <- 0.9
c_real <- 1.1

# Edades
x <- 0:20

# Función de Gompertz
gompertz <- function(x, lo, g, c) {
  lo * g^((c^x) - 1)
}

# Generar datos con ruido aleatorio
lx_obs <- gompertz(x, lo_real, g_real, c_real) * rnorm(length(x), 1, 0.05)

# Crear dataframe de datos
datos <- data.frame(x = x, lx = lx_obs)

# Visualizar datos generados
plot(datos$x, datos$lx, type = "b", pch = 16, col = "blue",
     xlab = "Edad (x)", ylab = "lx", 
     main = "Datos de supervivencia - Modelo de Gompertz")

# FUNCIÓN PARA ESTIMAR PARÁMETROS
estimar_gompertz <- function(x, lx_obs, lo_fijo = NULL) {
  
  # Función objetivo para minimizar
  funcion_objetivo <- function(params) {
    if (is.null(lo_fijo)) {
      lo <- params[1]
      g <- params[2]
      c <- params[3]
    } else {
      lo <- lo_fijo
      g <- params[1]
      c <- params[2]
    }
    
    lx_pred <- lo * g^((c^x) - 1)
    sum((lx_obs - lx_pred)^2) # Suma de cuadrados de residuos
  }
  
  # Valores iniciales para la optimización
  if (is.null(lo_fijo)) {
    # Estimar lo, g, c
    iniciales <- c(lo = mean(lx_obs), g = 0.95, c = 1.05)
    resultado <- optim(iniciales, funcion_objetivo, method = "L-BFGS-B",
                       lower = c(0.1, 0.1, 1.001), 
                       upper = c(10000, 0.999, 1.5))
    parametros <- resultado$par
    names(parametros) <- c("lo", "g", "c")
  } else {
    # Estimar solo g y c (lo fijo)
    iniciales <- c(g = 0.95, c = 1.05)
    resultado <- optim(iniciales, funcion_objetivo, method = "L-BFGS-B",
                       lower = c(0.1, 1.001), 
                       upper = c(0.999, 1.5))
    parametros <- c(lo = lo_fijo, resultado$par)
  }
  
  return(list(parametros = parametros, 
              convergencia = resultado$convergence,
              valor_objetivo = resultado$value))
}

# ESTIMACIÓN 1: Estimar los tres parámetros (lo, g, c)
cat("=== ESTIMACIÓN DE LOS TRES PARÁMETROS ===\n")
estimacion1 <- estimar_gompertz(datos$x, datos$lx)
print(estimacion1$parametros)

# ESTIMACIÓN 2: Estimar solo g y c (asumiendo lo conocido)
cat("\n=== ESTIMACIÓN DE g y c (lo conocido) ===\n")
estimacion2 <- estimar_gompertz(datos$x, datos$lx, lo_fijo = lo_real)
print(estimacion2$parametros)

# Comparar con valores reales
cat("\n=== COMPARACIÓN CON VALORES REALES ===\n")
cat("Valores reales: lo =", lo_real, "g =", g_real, "c =", c_real, "\n")

# Calcular predicciones
lx_pred1 <- gompertz(datos$x, estimacion1$parametros["lo"], 
                     estimacion1$parametros["g"], estimacion1$parametros["c"])
lx_pred2 <- gompertz(datos$x, estimacion2$parametros["lo"], 
                     estimacion2$parametros["g"], estimacion2$parametros["c"])

# Visualizar resultados
plot(datos$x, datos$lx, type = "b", pch = 16, col = "blue",
     xlab = "Edad (x)", ylab = "lx", 
     main = "Ajuste del Modelo de Gompertz",
     ylim = range(c(datos$lx, lx_pred1, lx_pred2)))

lines(datos$x, lx_pred1, type = "l", col = "red", lwd = 2)
lines(datos$x, lx_pred2, type = "l", col = "green", lwd = 2)

legend("topright", 
       legend = c("Datos observados", 
                  "Ajuste 3 parámetros", 
                  "Ajuste g y c (lo conocido)"),
       col = c("blue", "red", "green"), 
       pch = c(16, NA, NA), lty = c(1, 1, 1))

# Calcular medidas de bondad de ajuste
calcular_r2 <- function(obs, pred) {
  1 - sum((obs - pred)^2) / sum((obs - mean(obs))^2)
}

r2_1 <- calcular_r2(datos$lx, lx_pred1)
r2_2 <- calcular_r2(datos$lx, lx_pred2)

cat("\n=== BONDAD DE AJUSTE (R²) ===\n")
cat("R² ajuste 3 parámetros:", round(r2_1, 4), "\n")
cat("R² ajuste g y c:", round(r2_2, 4), "\n")

# Función para usar con datos propios
ajustar_gompertz_datos_propios <- function(x, lx, lo_conocido = NULL) {
  cat("\n=== AJUSTE CON DATOS PROPIOS ===\n")
  
  if (is.null(lo_conocido)) {
    resultado <- estimar_gompertz(x, lx)
    cat("Parámetros estimados:\n")
    print(resultado$parametros)
  } else {
    resultado <- estimar_gompertz(x, lx, lo_fijo = lo_conocido)
    cat("Parámetros estimados (lo fijo =", lo_conocido, "):\n")
    print(resultado$parametros)
  }
  
  return(resultado)
}

# EJEMPLO DE USO CON DATOS PROPIOS
# Descomenta y modifica las siguientes líneas para usar tus propios datos:

# x_propio <- c(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10)
# lx_propio <- c(1000, 850, 720, 610, 520, 440, 370, 310, 260, 220, 185)
# resultado_propio <- ajustar_gompertz_datos_propios(x_propio, lx_propio)