library(shiny)
library(ggplot2)
library(dplyr)
library(DT)

# Función corregida: lx = lo * g^((c^x) - 1)
lx_function <- function(x, lo, g, c) {
  lo * g^((c^x) - 1)
}

# UI
ui <- fluidPage(
  titlePanel("Modelo de Supervivencia de Gompertz lx = lo * g^((c^x) - 1)"),
  
  sidebarLayout(
    sidebarPanel(
      h4("Parámetros del Modelo"),
      numericInput("lo_param", "Valor inicial lo (cohorte):", 
                   value = 100000, min = 1000, max = 1000000, step = 1000),
      numericInput("g_param", "Parámetro g:", 
                   value = 0.95, min = 0.5, max = 0.999, step = 0.001),
      numericInput("c_param", "Parámetro c:", 
                   value = 1.1, min = 1.001, max = 1.5, step = 0.001),
      
      hr(),
      h4("Rango de Edades"),
      numericInput("x_min", "Edad mínima (x):", value = 0, min = 0),
      numericInput("x_max", "Edad máxima (x):", value = 100, min = 1),
      numericInput("x_step", "Incremento de edad:", value = 1, min = 1, max = 5),
      
      hr(),
      h4("Cálculo en Edad Específica"),
      numericInput("x_calc", "Edad para cálculo:", value = 65, min = 0),
      actionButton("calculate", "Calcular Valores"),
      
      hr(),
      downloadButton("downloadData", "Descargar Tabla de Vida")
    ),
    
    mainPanel(
      tabsetPanel(
        tabPanel("Gráficas Principales",
                 fluidRow(
                   column(6, plotOutput("lx_plot")),
                   column(6, plotOutput("qx_plot"))
                 ),
                 fluidRow(
                   column(6, plotOutput("dx_plot")),
                   column(6, plotOutput("log_lx_plot"))
                 )
        ),
        
        tabPanel("Tabla de Vida",
                 h4("Valores en edad específica:"),
                 tableOutput("specific_values"),
                 hr(),
                 h4("Tabla de Vida Completa"),
                 DTOutput("life_table")
        ),
        
        tabPanel("Indicadores",
                 h4("Indicadores de Supervivencia"),
                 verbatimTextOutput("survival_indicators"),
                 hr(),
                 h4("Esperanza de Vida"),
                 plotOutput("ex_plot")
        ),
        
        tabPanel("Análisis del Modelo",
                 h4("Comportamiento de la Función"),
                 verbatimTextOutput("model_analysis"),
                 hr(),
                 h4("Verificación de Cálculos"),
                 tableOutput("verification_table")
        )
      )
    )
  ) 
)

# Server
server <- function(input, output, session) {
  
  # Generar datos de la tabla de vida
  life_table_data <- reactive({
    ages <- seq(input$x_min, input$x_max, by = input$x_step)
    n <- length(ages)
    
    # Calcular lx con la fórmula corregida
    lx_values <- lx_function(ages, input$lo_param, input$g_param, input$c_param)
    
    # Crear data frame base
    df <- data.frame(
      x = ages,
      lx = lx_values,
      stringsAsFactors = FALSE
    )
    
    # Calcular dx (defunciones entre x y x+1)
    df$dx <- c(-diff(df$lx), NA)
    
    # Calcular qx (probabilidad de muerte)
    df$qx <- c(df$dx[-n] / df$lx[-n], NA)
    
    # Calcular px (probabilidad de supervivencia)
    df$px <- 1 - df$qx
    
    # Calcular Lx (años-persona vividos)
    df$Lx <- c((df$lx[-n] + df$lx[-1]) / 2, df$lx[n] / 2)
    
    # Calcular Tx (años-persona por vivir)
    df$Tx <- NA
    df$Tx[n] <- df$Lx[n]
    for(i in (n-1):1) {
      df$Tx[i] <- df$Lx[i] + df$Tx[i + 1]
    }
    
    # Calcular ex (esperanza de vida)
    df$ex <- df$Tx / df$lx
    
    return(df)
  })
  
  # Gráfica de lx (supervivientes)
  output$lx_plot <- renderPlot({
    df <- life_table_data()
    ggplot(df, aes(x = x, y = lx)) +
      geom_line(color = "blue", size = 1) +
      geom_point(color = "blue", size = 1) +
      labs(title = "Supervivientes lx",
           subtitle = paste("lx =", input$lo_param, "*", input$g_param, "^(", input$c_param, "^x - 1)"),
           x = "Edad (x)", y = "lx") +
      theme_minimal() +
      scale_y_continuous(labels = scales::comma)
  })
  
  # Gráfica de qx (probabilidad de muerte)
  output$qx_plot <- renderPlot({
    df <- life_table_data() %>% filter(!is.na(qx))
    if(nrow(df) > 0) {
      ggplot(df, aes(x = x, y = qx)) +
        geom_line(color = "red", size = 1) +
        geom_point(color = "red", size = 1) +
        labs(title = "Probabilidad de Muerte qx",
             x = "Edad (x)", y = "qx") +
        theme_minimal() +
        scale_y_continuous(labels = scales::percent)
    }
  })
  
  # Gráfica de dx (defunciones)
  output$dx_plot <- renderPlot({
    df <- life_table_data() %>% filter(!is.na(dx))
    if(nrow(df) > 0) {
      ggplot(df, aes(x = x, y = dx)) +
        geom_col(fill = "darkred", alpha = 0.7) +
        labs(title = "Defunciones dx",
             x = "Edad (x)", y = "dx") +
        theme_minimal() +
        scale_y_continuous(labels = scales::comma)
    }
  })
  
  # Gráfica de log(lx)
  output$log_lx_plot <- renderPlot({
    df <- life_table_data()
    ggplot(df, aes(x = x, y = log(lx))) +
      geom_line(color = "purple", size = 1) +
      geom_point(color = "purple", size = 1) +
      labs(title = "Logaritmo de Supervivientes log(lx)",
           x = "Edad (x)", y = "log(lx)") +
      theme_minimal()
  })
  
  # Valores en edad específica
  specific_values <- eventReactive(input$calculate, {
    x_val <- input$x_calc
    df <- life_table_data() %>% filter(x == x_val)
    
    if(nrow(df) > 0) {
      data.frame(
        Indicador = c("Edad", "Supervivientes lx", "Prob. Muerte qx", 
                      "Defunciones dx", "Prob. Supervivencia px", "Esperanza Vida ex"),
        Valor = c(
          df$x,
          round(df$lx, 1),
          ifelse(is.na(df$qx), NA, round(df$qx, 4)),
          ifelse(is.na(df$dx), NA, round(df$dx, 1)),
          ifelse(is.na(df$px), NA, round(df$px, 4)),
          round(df$ex, 2)
        )
      )
    }
  })
  
  output$specific_values <- renderTable({
    specific_values()
  })
  
  # Tabla de vida completa
  output$life_table <- renderDT({
    df <- life_table_data()
    datatable(df %>% select(x, lx, dx, qx, px, ex),
              options = list(
                pageLength = 20,
                scrollX = TRUE,
                dom = 'tip'
              ),
              rownames = FALSE,
              colnames = c('Edad x', 'lx', 'dx', 'qx', 'px', 'ex')) %>%
      formatRound(columns = c('lx', 'dx'), digits = 1) %>%
      formatRound(columns = c('qx', 'px'), digits = 4) %>%
      formatRound(columns = 'ex', digits = 2)
  })
  
  # Indicadores de supervivencia
  output$survival_indicators <- renderPrint({
    df <- life_table_data()
    
    # Encontrar edades características
    lx_50 <- df %>% filter(lx <= input$lo_param * 0.5) %>% slice(1)
    lx_10 <- df %>% filter(lx <= input$lo_param * 0.1) %>% slice(1)
    lx_25 <- df %>% filter(lx <= input$lo_param * 0.25) %>% slice(1)
    lx_75 <- df %>% filter(lx <= input$lo_param * 0.75) %>% slice(1)
    
    cat("INDICADORES DEL MODELO\n")
    cat("=====================\n")
    cat("Fórmula: lx = lo * g^((c^x) - 1)\n\n")
    
    cat("Parámetros:\n")
    cat("lo =", format(input$lo_param, big.mark = ","), "\n")
    cat("g =", input$g_param, "\n")
    cat("c =", input$c_param, "\n\n")
    
    cat("Supervivencia:\n")
    cat("Edad mediana de supervivencia:", ifelse(nrow(lx_50) > 0, lx_50$x, "No alcanzada"), "\n")
    cat("Edad del 25% superviviente:", ifelse(nrow(lx_25) > 0, lx_25$x, "No alcanzada"), "\n")
    cat("Edad del 10% superviviente:", ifelse(nrow(lx_10) > 0, lx_10$x, "No alcanzada"), "\n")
    cat("Edad del 75% superviviente:", ifelse(nrow(lx_75) > 0, lx_75$x, "No alcanzada"), "\n")
    
    cat("\nMortalidad:\n")
    max_qx_row <- df %>% filter(!is.na(qx)) %>% arrange(desc(qx)) %>% slice(1)
    if(nrow(max_qx_row) > 0) {
      cat("Máxima probabilidad de muerte:", round(max_qx_row$qx, 4), "a edad", max_qx_row$x, "\n")
    }
    
    cat("\nEsperanza de vida:\n")
    cat("Esperanza de vida al nacer:", round(df$ex[1], 2), "años\n")
    e65 <- df %>% filter(x == 65)
    if(nrow(e65) > 0 && !is.na(e65$ex)) {
      cat("Esperanza de vida a los 65 años:", round(e65$ex, 2), "años\n")
    }
  })
  
  # Gráfica de esperanza de vida
  output$ex_plot <- renderPlot({
    df <- life_table_data() %>% filter(!is.na(ex))
    if(nrow(df) > 0) {
      ggplot(df, aes(x = x, y = ex)) +
        geom_line(color = "darkgreen", size = 1) +
        geom_point(color = "darkgreen", size = 1) +
        labs(title = "Esperanza de Vida ex",
             x = "Edad (x)", y = "Esperanza de Vida (años)") +
        theme_minimal()
    }
  })
  
  # Análisis del modelo
  output$model_analysis <- renderPrint({
    df <- life_table_data()
    
    cat("ANÁLISIS DEL MODELO lx = lo * g^((c^x) - 1)\n")
    cat("==========================================\n\n")
    
    cat("Comportamiento de los parámetros:\n")
    cat("- g < 1: La supervivencia disminuye con la edad\n")
    cat("- g más cercano a 1: Mortalidad más baja\n")
    cat("- g más pequeño: Mortalidad más alta\n")
    cat("- c > 1: La mortalidad aumenta exponencialmente con la edad\n")
    cat("- c más grande: Curva de mortalidad más pronunciada\n\n")
    
    cat("Verificación de cálculos para x = 0:\n")
    lx0_calculado <- lx_function(0, input$lo_param, input$g_param, input$c_param)
    cat("lx(0) calculado:", round(lx0_calculado, 2), "\n")
    cat("lx(0) teórico:", input$lo_param, "*", input$g_param, "^(", input$c_param, "^0 - 1) =", 
        input$lo_param, "*", input$g_param, "^(0) =", input$lo_param, "\n")
    
    cat("\nVerificación para x = 1:\n")
    lx1_calculado <- lx_function(1, input$lo_param, input$g_param, input$c_param)
    cat("lx(1) calculado:", round(lx1_calculado, 2), "\n")
    cat("lx(1) teórico:", input$lo_param, "*", input$g_param, "^(", input$c_param, "^1 - 1) =", 
        input$lo_param, "*", input$g_param, "^(", round(input$c_param - 1, 3), ") =", 
        round(input$lo_param * input$g_param^(input$c_param - 1), 2), "\n")
  })
  
  # Tabla de verificación
  output$verification_table <- renderTable({
    ages <- c(0, 1, 5, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100)
    ages <- ages[ages >= input$x_min & ages <= input$x_max]
    
    data.frame(
      Edad = ages,
      lx_calculado = round(lx_function(ages, input$lo_param, input$g_param, input$c_param), 1),
      Exponente = round((input$c_param^ages) - 1, 4),
      g_exponente = round(input$g_param^((input$c_param^ages) - 1), 6)
    )
  })
  
  # Descargar datos
  output$downloadData <- downloadHandler(
    filename = function() {
      paste("tabla_vida_modelo_", Sys.Date(), ".csv", sep = "")
    },
    content = function(file) {
      write.csv(life_table_data(), file, row.names = FALSE, na = "")
    }
  )
}

# Ejecutar la aplicación
shinyApp(ui = ui, server = server)