library(shiny)
library(ggplot2)
library(dplyr)
library(plotly)

# Definir la interfaz de usuario
ui <- fluidPage(
  titlePanel("Modelo de Supervivencia de Sang"),
  sidebarLayout(
    sidebarPanel(
      h4("Parámetros del Modelo"),
      numericInput("lo", "Tamaño de Cohorte Inicial (l₀):", 
                   value = 1000, min = 1, max = 100000),
      numericInput("w", "Infinito Actuarial (ω):", 
                   value = 100, min = 50, max = 150),
      numericInput("b", "Parámetro b:", 
                   value = 0.95, min = 0.8, max = 0.999, step = 0.001),
      sliderInput("x_range", "Rango de edades a visualizar:",
                  min = 0, max = 150, value = c(0, 100)),
      actionButton("calcular", "Calcular", class = "btn-primary"),
      br(), br(),
      h4("Valores de Ejemplo Rápidos"),
      actionButton("ejemplo1", "Ejemplo 1: l₀=1000, ω=100, b=0.95"),
      actionButton("ejemplo2", "Ejemplo 2: l₀=10000, ω=120, b=0.999")
    ),
    
    mainPanel(
      tabsetPanel(
        tabPanel("Gráfica de Supervivencia",
                 plotlyOutput("grafica_supervivencia"),
                 br(),
                 h4("Ecuación del Modelo"),
                 withMathJax("$$l_x = \\frac{l_0}{1 - b^{\\omega}} \\cdot (b^x - b^{\\omega})$$")),
        
        tabPanel("Tabla de Vida",
                 downloadButton("descargar", "Descargar Tabla"),
                 br(), br(),
                 dataTableOutput("tabla_vida")),
        
        tabPanel("Análisis",
                 fluidRow(
                   column(6,
                          h4("Estadísticas Descriptivas"),
                          tableOutput("estadisticas")),
                   column(6,
                          h4("Probabilidades de Supervivencia"),
                          tableOutput("probabilidades"))
                 )),
        
        tabPanel("Información del Modelo",
                 h3("Modelo de Supervivencia de Sang"),
                 p("Este modelo utiliza la función:"),
                 withMathJax("$$l_x = \\frac{l_0}{1 - b^{\\omega}} \\cdot (b^x - b^{\\omega})$$"),
                 p("Donde:"),
                 tags$ul(
                   tags$li(withMathJax("$l_x$: Número de supervivientes a la edad $x$")),
                   tags$li(withMathJax("$l_0$: Tamaño inicial de la cohorte")),
                   tags$li(withMathJax("$\\omega$: Edad máxima (infinito actuarial)")),
                   tags$li(withMathJax("$b$: Parámetro de decrecimiento ($b < 1$)"))
                 ),
                 p("Características del modelo:"),
                 tags$ul(
                   tags$li("La función es decreciente con la edad"),
                   tags$li("lₓ = 0 cuando x = ω"),
                   tags$li("El parámetro b controla la forma de la curva de supervivencia")
                 ))
      )
    )
  )
)

# Definir el servidor
server <- function(input, output, session) {
  
  # Reactive values para almacenar resultados
  valores <- reactiveValues(datos = NULL)
  
  # Función del modelo de Sang
  modelo_sang <- function(x, lo, w, b) {
    if (x >= w) {
      return(0)
    }
    (lo / (1 - b^w)) * (b^x - b^w)
  }
  
  # Calcular cuando se presiona el botón
  observeEvent(input$calcular, {
    calcular_modelo()
  })
  
  # Ejemplos predefinidos
  observeEvent(input$ejemplo1, {
    updateNumericInput(session, "lo", value = 1000)
    updateNumericInput(session, "w", value = 100)
    updateNumericInput(session, "b", value = 0.95)
    calcular_modelo()
  })
  
  observeEvent(input$ejemplo2, {
    updateNumericInput(session, "lo", value = 10000)
    updateNumericInput(session, "w", value = 120)
    updateNumericInput(session, "b", value = 0.999)
    calcular_modelo()
  })
  
  # Función para calcular el modelo
  calcular_modelo <- function() {
    req(input$lo, input$w, input$b)
    
    # Validar parámetros
    if (input$b >= 1) {
      showNotification("El parámetro b debe ser menor que 1", type = "error")
      return()
    }
    
    # Crear secuencia de edades
    edades <- seq(0, input$w, by = 1)
    
    # Calcular lx para cada edad
    lx <- sapply(edades, modelo_sang, lo = input$lo, w = input$w, b = input$b)
    
    # Crear data frame con resultados
    datos <- data.frame(
      Edad = edades,
      lx = lx,
      dx = c(-diff(lx), 0),  # Defunciones entre x y x+1
      qx = c(ifelse(lx[-length(lx)] > 0, 
                    -diff(lx)/lx[-length(lx)], 0), 0)  # Probabilidad de muerte
    )
    
    # Calcular probabilidad de supervivencia
    datos$px <- 1 - datos$qx
    
    valores$datos <- datos
  }
  
  # Gráfica de supervivencia
  output$grafica_supervivencia <- renderPlotly({
    req(valores$datos)
    
    datos_filtrados <- valores$datos %>%
      filter(Edad >= input$x_range[1], Edad <= input$x_range[2])
    
    p <- ggplot(datos_filtrados, aes(x = Edad, y = lx)) +
      geom_line(color = "steelblue", size = 1) +
      geom_point(color = "steelblue", size = 0.5) +
      labs(title = "Curva de Supervivencia - Modelo de Sang",
           x = "Edad (x)",
           y = "Supervivientes (lₓ)") +
      theme_minimal() +
      theme(plot.title = element_text(hjust = 0.5))
    
    ggplotly(p) %>%
      layout(hovermode = 'x unified')
  })
  
  # Tabla de vida
  output$tabla_vida <- renderDataTable({
    req(valores$datos)
    
    valores$datos %>%
      mutate(across(c(lx, dx), round, 2),
             across(c(qx, px), round, 4))
  }, options = list(pageLength = 10))
  
  # Estadísticas descriptivas
  output$estadisticas <- renderTable({
    req(valores$datos)
    
    datos <- valores$datos
    
    # Esperanza de vida aproximada
    esperanza_vida <- sum(datos$lx[-1]) / datos$lx[1]
    
    # Edad modal al fallecimiento
    edad_modal <- datos$Edad[which.max(datos$dx)]
    
    data.frame(
      Estadística = c("Cohorte Inicial", 
                      "Edad Máxima (ω)",
                      "Esperanza de Vida",
                      "Edad Modal al Fallecimiento",
                      "Supervivientes a 65 años"),
      Valor = c(input$lo,
                input$w,
                round(esperanza_vida, 2),
                edad_modal,
                round(datos$lx[datos$Edad == 65][1], 2))
    )
  }, bordered = TRUE)
  
  # Probabilidades de supervivencia
  output$probabilidades <- renderTable({
    req(valores$datos)
    
    datos <- valores$datos
    
    # Encontrar valores para edades específicas
    lx_20 <- datos$lx[datos$Edad == 20][1]
    lx_65 <- datos$lx[datos$Edad == 65][1]
    
    data.frame(
      Probabilidad = c("20p0 (Llegar a 20 años)",
                       "45p20 (Llegar a 65 desde 20)",
                       "65p0 (Llegar a 65 años)"),
      Valor = c(ifelse(!is.na(lx_20), 
                       round(lx_20/input$lo, 4), NA),
                ifelse(!is.na(lx_20) & !is.na(lx_65) & lx_20 > 0,
                       round(lx_65/lx_20, 4), NA),
                ifelse(!is.na(lx_65), 
                       round(lx_65/input$lo, 4), NA))
    )
  }, bordered = TRUE)
  
  # Descargar tabla
  output$descargar <- downloadHandler(
    filename = function() {
      paste0("tabla_vida_sang_", Sys.Date(), ".csv")
    },
    content = function(file) {
      write.csv(valores$datos, file, row.names = FALSE)
    }
  )
}

# Ejecutar la aplicación
shinyApp(ui = ui, server = server)