library(shiny)
library(ggplot2)
library(gridExtra)

ui <- fluidPage(
  titlePanel("Análisis del modelo de Moivre con  w y x(edad)"),
  
  sidebarLayout(
    sidebarPanel(
      sliderInput("w", "Valor de w:", 
                  min = 1, max = 110, value = 100, step = 0.1),
      sliderInput("n", "edad:", 
                  min = 1, max = 110, value = 20, step = 0.1),
      
      #numericInput("n", "Edad:", 
                   #value = 10, min = 0, max =110, step = 1),
      
     #numericInput("s", "Número de puntos:", 
                 # value = 2, min = 0.8, max = 0.999999),
      
      selectInput("funcion", "Modelo :",
                  choices = c("Moivre" = "moivre"))
                              
                   # "dormoy1" = "dormoy1",
                             # "Exponencial" = "exponencial",
                             # "Seno" = "seno"))
     
    ),
    
    mainPanel(
      plotOutput("graficos_combinados"),
      verbatimTextOutput("resultados_completos")
      
  )
)
)

server <- function(input, output) {
  
  calcular_funcion <- function(w, x, moivre) {
    switch(moivre,
           "dormoy1" = s^n,
           "moivre" = 1-x/w,
          "exponencial" = exp(w * x),
          "seno" = sin(w * x))
  }
  
  calcular_mx <- function(w, x) {
    if (w == 0) return(NA)
    return(1/(w - x))
  }
  calcular_px <- function(w,n) {
    if (w == 0) return(NA)
    return((w-n-1)/(w - n))
  }
  calcular_ex <- function(w,n) {
    if (w == 0) return(NA)
    return((w-n)/2)
  }
  datos_reactivos <- reactive({
    #num_puntos<-100
    x <- seq(0, input$w, length.out = 1000)
    y <- calcular_funcion(input$w, x, input$funcion)
    mx <- calcular_mx(input$w, x)
    px <- calcular_px(input$w, x)
    ex<-calcular_ex(input$w,x)
    data.frame(x = x, y = y, mx = mx, px=px, ex=ex)
  })
  
  output$graficos_combinados <- renderPlot({
    datos <- datos_reactivos()
    w <- input$w
    n <- input$n
    n_valido <- n >= 0 & n <= w
    
    # Gráfico de la función principal
    p1 <- ggplot(datos, aes(x = x, y = y)) +
      geom_line(color = "darkblue", size = 1) +
      labs(title = paste("f(x) con w =", w),
           x = "x", y = "f(x)") +
      theme_minimal()
    
    # Gráfico de mx
    p2 <- ggplot(datos, aes(x = x, y = mx)) +
      geom_line(color = "darkred", size = 1) +
      labs(title = "Tanto x = 1/(w - x)",
           x = "x", y = "mx") +
      theme_minimal()
    # Gráfico de px
    p3 <- ggplot(datos, aes(x = x, y = px)) +
      geom_line(color = "darkred", size = 1) +
      labs(title = "px = (w-x-1)/(w - x)",
           x = "x", y = "px") +
      theme_minimal()
    
    p4 <- ggplot(datos, aes(x = x, y = ex)) +
      geom_line(color = "darkred", size = 1) +
      labs(title = "esperanza = (w-x)/2",
           x = "x", y = "ex") +
      theme_minimal()
    
    # Agregar puntos en n si es válido
    if (n_valido) {
      y_n <- calcular_funcion(w, n, input$funcion)
      mx_n <- calcular_mx(w, n)
      px_n <- calcular_px(w, n)
      ex_n <- calcular_ex(w, n)
      
      
      p1 <- p1 + 
        geom_point(aes(x = n, y = y_n), color = "red", size = 3) +
        geom_vline(xintercept = n, linetype = "dashed", color = "red")
      
      p2 <- p2 + 
        geom_point(aes(x = n, y = mx_n), color = "red", size = 3) +
        geom_vline(xintercept = n, linetype = "dashed", color = "red")
      p3 <- p3 + 
        geom_point(aes(x = n, y = px_n), color = "red", size = 3) +
        geom_vline(xintercept = n, linetype = "dashed", color = "red")
      p4 <- p4 + 
        geom_point(aes(x = n, y = ex_n), color = "red", size = 3) +
        geom_vline(xintercept = n, linetype = "dashed", color = "red")
    }
    
    grid.arrange(p1, p2, p3, p4 , ncol = 2)
  })
  
  output$resultados_completos <- renderPrint({
    datos <- datos_reactivos()
    w <- input$w
    n <- input$n
    
    valor_funcion_n <- calcular_funcion(w, n, input$funcion)
    valor_mx_n <- calcular_mx(w, n)
    valor_px_n <- calcular_px(w, n)
    valor_ex_n<-calcular_ex(w,n)
    n_valido <- n >= 0 & n <= w
    
    cat("RESULTADOS PARA w =", w, "y x =", n, "\n")
    cat("=====================================\n\n")
    
    cat("EN x = n:\n")
    cat("---------\n")
    cat("f(n) =", round(valor_funcion_n, 6), "\n")
    cat("Tanto x = 1/(w - x) =", round(valor_mx_n, 6), "\n")
    cat("px = w-x-1/(w - x) =", round(valor_px_n, 6), "\n")
    cat("eesperanza de vida = w-x/2 =", round(valor_ex_n, 6), "\n")
    if (!n_valido) cat("⚠️ Nota: n fuera del rango [0, w]\n")
    
    cat("Juan\n")
   
    
  })
}

shinyApp(ui = ui, server = server)